<?php
require_once '../includes/auth.php';
require_once '../config/database.php';

requireAdmin();

$database = new Database();
$db = $database->getConnection();

// Get analytics data
$analytics_query = "
    SELECT 
        COUNT(DISTINCT r.id) as total_reservations,
        COUNT(DISTINCT u.id) as total_customers,
        AVG(rev.rating) as avg_rating,
        COUNT(DISTINCT rev.id) as total_reviews,
        COUNT(CASE WHEN r.reservation_date = CURDATE() THEN 1 END) as today_reservations,
        COUNT(CASE WHEN r.reservation_date >= CURDATE() - INTERVAL 7 DAY THEN 1 END) as week_reservations,
        COUNT(CASE WHEN r.reservation_date >= CURDATE() - INTERVAL 30 DAY THEN 1 END) as month_reservations
    FROM reservations r
    LEFT JOIN users u ON r.customer_id = u.id
    LEFT JOIN reviews rev ON rev.customer_id = u.id
";
$analytics_stmt = $db->prepare($analytics_query);
$analytics_stmt->execute();
$analytics = $analytics_stmt->fetch(PDO::FETCH_ASSOC);

// Get monthly booking trends
$monthly_trends_query = "
    SELECT 
        DATE_FORMAT(reservation_date, '%Y-%m') as month,
        COUNT(*) as bookings
    FROM reservations 
    WHERE reservation_date >= DATE_SUB(CURDATE(), INTERVAL 6 MONTH)
    GROUP BY DATE_FORMAT(reservation_date, '%Y-%m')
    ORDER BY month ASC
";
$monthly_stmt = $db->prepare($monthly_trends_query);
$monthly_stmt->execute();
$monthly_trends = $monthly_stmt->fetchAll(PDO::FETCH_ASSOC);

// Get peak hours
$peak_hours_query = "
    SELECT 
        HOUR(reservation_time) as hour,
        COUNT(*) as bookings
    FROM reservations 
    WHERE reservation_date >= DATE_SUB(CURDATE(), INTERVAL 30 DAY)
    GROUP BY HOUR(reservation_time)
    ORDER BY hour ASC
";
$peak_stmt = $db->prepare($peak_hours_query);
$peak_stmt->execute();
$peak_hours = $peak_stmt->fetchAll(PDO::FETCH_ASSOC);

// Get review ratings distribution
$ratings_query = "
    SELECT 
        rating,
        COUNT(*) as count
    FROM reviews 
    GROUP BY rating
    ORDER BY rating DESC
";
$ratings_stmt = $db->prepare($ratings_query);
$ratings_stmt->execute();
$ratings_distribution = $ratings_stmt->fetchAll(PDO::FETCH_ASSOC);

// Get table utilization
$table_utilization_query = "
    SELECT 
        rt.table_number,
        rt.location,
        COUNT(r.id) as total_bookings,
        ROUND((COUNT(r.id) / 30.0) * 100, 1) as utilization_rate
    FROM restaurant_tables rt
    LEFT JOIN reservations r ON rt.id = r.table_id 
        AND r.reservation_date >= DATE_SUB(CURDATE(), INTERVAL 30 DAY)
    GROUP BY rt.id, rt.table_number, rt.location
    ORDER BY utilization_rate DESC
";
$utilization_stmt = $db->prepare($table_utilization_query);
$utilization_stmt->execute();
$table_utilization = $utilization_stmt->fetchAll(PDO::FETCH_ASSOC);

$page_title = 'Analytics Dashboard';
include '../includes/header.php';
?>

<div class="flex h-screen bg-gray-50">
    <?php include '../includes/sidebar.php'; ?>
    
    <div class="flex-1 flex flex-col overflow-hidden lg:ml-0">
        <?php include '../includes/dashboard_header.php'; ?>
        
        <main class="flex-1 overflow-y-auto p-6">
            <div class="space-y-6 fade-in">
                <div class="flex items-center justify-between">
                    <h1 class="text-2xl font-bold text-gray-900">Analytics Dashboard</h1>
                    <div class="flex space-x-2">
                        <select class="px-3 py-2 border border-gray-300 rounded-lg focus:ring-amber-500 focus:border-amber-500">
                            <option>Last 30 days</option>
                            <option>Last 3 months</option>
                            <option>Last 6 months</option>
                            <option>Last year</option>
                        </select>
                    </div>
                </div>

                <!-- KPI Cards -->
                <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6">
                    <div class="bg-white rounded-lg shadow-sm border border-gray-200 p-6 hover-scale">
                        <div class="flex items-center justify-between">
                            <div>
                                <p class="text-sm font-medium text-gray-600">Total Reservations</p>
                                <p class="text-2xl font-bold text-gray-900"><?php echo $analytics['total_reservations'] ?: '0'; ?></p>
                                <div class="flex items-center mt-1">
                                    <i data-lucide="trending-up" class="w-4 h-4 text-green-500 mr-1"></i>
                                    <span class="text-sm text-green-600">All time</span>
                                </div>
                            </div>
                            <i data-lucide="calendar" class="w-8 h-8 text-blue-600"></i>
                        </div>
                    </div>

                    <div class="bg-white rounded-lg shadow-sm border border-gray-200 p-6 hover-scale">
                        <div class="flex items-center justify-between">
                            <div>
                                <p class="text-sm font-medium text-gray-600">Total Customers</p>
                                <p class="text-2xl font-bold text-gray-900"><?php echo $analytics['total_customers'] ?: '0'; ?></p>
                                <div class="flex items-center mt-1">
                                    <i data-lucide="trending-up" class="w-4 h-4 text-green-500 mr-1"></i>
                                    <span class="text-sm text-green-600">Registered</span>
                                </div>
                            </div>
                            <i data-lucide="users" class="w-8 h-8 text-green-600"></i>
                        </div>
                    </div>

                    <div class="bg-white rounded-lg shadow-sm border border-gray-200 p-6 hover-scale">
                        <div class="flex items-center justify-between">
                            <div>
                                <p class="text-sm font-medium text-gray-600">Average Rating</p>
                                <p class="text-2xl font-bold text-gray-900"><?php echo $analytics['avg_rating'] ? number_format($analytics['avg_rating'], 1) : 'N/A'; ?></p>
                                <div class="flex items-center mt-1">
                                    <i data-lucide="star" class="w-4 h-4 text-yellow-500 mr-1"></i>
                                    <span class="text-sm text-yellow-600">Customer reviews</span>
                                </div>
                            </div>
                            <i data-lucide="star" class="w-8 h-8 text-yellow-600"></i>
                        </div>
                    </div>

                    <div class="bg-white rounded-lg shadow-sm border border-gray-200 p-6 hover-scale">
                        <div class="flex items-center justify-between">
                            <div>
                                <p class="text-sm font-medium text-gray-600">Today's Bookings</p>
                                <p class="text-2xl font-bold text-gray-900"><?php echo $analytics['today_reservations'] ?: '0'; ?></p>
                                <div class="flex items-center mt-1">
                                    <i data-lucide="clock" class="w-4 h-4 text-purple-500 mr-1"></i>
                                    <span class="text-sm text-purple-600">Active today</span>
                                </div>
                            </div>
                            <i data-lucide="clock" class="w-8 h-8 text-purple-600"></i>
                        </div>
                    </div>
                </div>

                <!-- Charts Row 1 -->
                <div class="grid grid-cols-1 lg:grid-cols-2 gap-6">
                    <!-- Monthly Trends -->
                    <div class="bg-white rounded-lg shadow-sm border border-gray-200 p-6">
                        <h3 class="text-lg font-semibold text-gray-900 mb-4">Monthly Booking Trends</h3>
                        <div class="h-64 flex items-end justify-between space-x-2">
                            <?php if (!empty($monthly_trends)): ?>
                                <?php 
                                $max_bookings = max(array_column($monthly_trends, 'bookings'));
                                foreach ($monthly_trends as $trend): 
                                    $height = $max_bookings > 0 ? ($trend['bookings'] / $max_bookings) * 100 : 0;
                                    $month_name = date('M', strtotime($trend['month'] . '-01'));
                                ?>
                                    <div class="flex flex-col items-center flex-1">
                                        <div class="w-full bg-amber-500 rounded-t" style="height: <?php echo $height; ?>%; min-height: 4px;"></div>
                                        <div class="text-xs text-gray-600 mt-2"><?php echo $month_name; ?></div>
                                        <div class="text-xs font-medium text-gray-900"><?php echo $trend['bookings']; ?></div>
                                    </div>
                                <?php endforeach; ?>
                            <?php else: ?>
                                <div class="w-full text-center text-gray-500">No data available</div>
                            <?php endif; ?>
                        </div>
                    </div>

                    <!-- Peak Hours -->
                    <div class="bg-white rounded-lg shadow-sm border border-gray-200 p-6">
                        <h3 class="text-lg font-semibold text-gray-900 mb-4">Peak Booking Hours</h3>
                        <div class="h-64 flex items-end justify-between space-x-1">
                            <?php if (!empty($peak_hours)): ?>
                                <?php 
                                $max_hour_bookings = max(array_column($peak_hours, 'bookings'));
                                for ($hour = 10; $hour <= 22; $hour++): 
                                    $hour_data = array_filter($peak_hours, function($h) use ($hour) { return $h['hour'] == $hour; });
                                    $bookings = !empty($hour_data) ? reset($hour_data)['bookings'] : 0;
                                    $height = $max_hour_bookings > 0 ? ($bookings / $max_hour_bookings) * 100 : 0;
                                ?>
                                    <div class="flex flex-col items-center flex-1">
                                        <div class="w-full bg-green-500 rounded-t" style="height: <?php echo $height; ?>%; min-height: 4px;"></div>
                                        <div class="text-xs text-gray-600 mt-2"><?php echo $hour; ?>:00</div>
                                        <div class="text-xs font-medium text-gray-900"><?php echo $bookings; ?></div>
                                    </div>
                                <?php endfor; ?>
                            <?php else: ?>
                                <div class="w-full text-center text-gray-500">No data available</div>
                            <?php endif; ?>
                        </div>
                    </div>
                </div>

                <!-- Charts Row 2 -->
                <div class="grid grid-cols-1 lg:grid-cols-2 gap-6">
                    <!-- Review Ratings -->
                    <div class="bg-white rounded-lg shadow-sm border border-gray-200 p-6">
                        <h3 class="text-lg font-semibold text-gray-900 mb-4">Review Rating Distribution</h3>
                        <div class="space-y-3">
                            <?php if (!empty($ratings_distribution)): ?>
                                <?php 
                                $total_reviews = array_sum(array_column($ratings_distribution, 'count'));
                                foreach ($ratings_distribution as $rating): 
                                    $percentage = $total_reviews > 0 ? ($rating['count'] / $total_reviews) * 100 : 0;
                                ?>
                                    <div class="flex items-center">
                                        <div class="w-16 text-sm font-medium text-gray-700"><?php echo $rating['rating']; ?> stars</div>
                                        <div class="flex-1 mx-3">
                                            <div class="bg-gray-200 rounded-full h-2">
                                                <div class="bg-yellow-500 h-2 rounded-full" style="width: <?php echo $percentage; ?>%"></div>
                                            </div>
                                        </div>
                                        <div class="w-12 text-sm text-gray-600"><?php echo $rating['count']; ?></div>
                                    </div>
                                <?php endforeach; ?>
                            <?php else: ?>
                                <div class="text-center text-gray-500">No reviews yet</div>
                            <?php endif; ?>
                        </div>
                    </div>

                    <!-- Table Utilization -->
                    <div class="bg-white rounded-lg shadow-sm border border-gray-200 p-6">
                        <h3 class="text-lg font-semibold text-gray-900 mb-4">Table Utilization (Last 30 Days)</h3>
                        <div class="space-y-4">
                            <?php foreach ($table_utilization as $table): ?>
                                <div class="flex items-center justify-between">
                                    <div class="flex-1">
                                        <div class="flex justify-between text-sm">
                                            <span class="font-medium text-gray-900">Table <?php echo $table['table_number']; ?></span>
                                            <span class="text-gray-500"><?php echo $table['utilization_rate']; ?>% (<?php echo $table['total_bookings']; ?> bookings)</span>
                                        </div>
                                        <div class="mt-1 bg-gray-200 rounded-full h-2">
                                            <div class="bg-amber-500 h-2 rounded-full" style="width: <?php echo min($table['utilization_rate'], 100); ?>%"></div>
                                        </div>
                                        <div class="text-xs text-gray-500 mt-1"><?php echo htmlspecialchars($table['location']); ?></div>
                                    </div>
                                </div>
                            <?php endforeach; ?>
                        </div>
                    </div>
                </div>

                <!-- Customer Insights -->
                <div class="bg-white rounded-lg shadow-sm border border-gray-200 p-6">
                    <h3 class="text-lg font-semibold text-gray-900 mb-4">Customer Insights</h3>
                    <div class="grid grid-cols-1 md:grid-cols-3 gap-6">
                        <div class="text-center">
                            <div class="text-3xl font-bold text-amber-600"><?php echo $analytics['week_reservations'] ?: '0'; ?></div>
                            <div class="text-sm text-gray-500">This Week's Bookings</div>
                            <div class="text-xs text-green-600">Last 7 days</div>
                        </div>
                        <div class="text-center">
                            <div class="text-3xl font-bold text-blue-600"><?php echo $analytics['month_reservations'] ?: '0'; ?></div>
                            <div class="text-sm text-gray-500">This Month's Bookings</div>
                            <div class="text-xs text-blue-600">Last 30 days</div>
                        </div>
                        <div class="text-center">
                            <div class="text-3xl font-bold text-purple-600"><?php echo $analytics['total_reviews'] ?: '0'; ?></div>
                            <div class="text-sm text-gray-500">Total Reviews</div>
                            <div class="text-xs text-purple-600">All time feedback</div>
                        </div>
                    </div>
                </div>
            </div>
        </main>
    </div>
</div>

<?php include '../includes/footer.php'; ?>