<?php
require_once '../includes/auth.php';
require_once '../config/database.php';

requireAdmin();

$database = new Database();
$db = $database->getConnection();

// Get filter parameters
$search = $_GET['search'] ?? '';
$sort_by = $_GET['sort_by'] ?? 'name';

// Build query with filters
$where_clause = "WHERE u.role = 'customer'";
$params = [];

if (!empty($search)) {
    $where_clause .= " AND (u.name LIKE :search OR u.email LIKE :search)";
    $params[':search'] = '%' . $search . '%';
}

$order_clause = "ORDER BY ";
switch ($sort_by) {
    case 'visits':
        $order_clause .= "total_reservations DESC";
        break;
    case 'spent':
        $order_clause .= "u.created_at DESC"; // Placeholder for spending data
        break;
    case 'lastVisit':
        $order_clause .= "last_visit DESC";
        break;
    default:
        $order_clause .= "u.name ASC";
}

// Get customers with their statistics
$customers_query = "
    SELECT u.*,
           COUNT(DISTINCT r.id) as total_reservations,
           MAX(r.reservation_date) as last_visit,
           AVG(rev.rating) as average_rating,
           COUNT(DISTINCT rev.id) as total_reviews
    FROM users u
    LEFT JOIN reservations r ON u.id = r.customer_id
    LEFT JOIN reviews rev ON u.id = rev.customer_id
    $where_clause
    GROUP BY u.id
    $order_clause
";

$customers_stmt = $db->prepare($customers_query);
foreach ($params as $key => $value) {
    $customers_stmt->bindValue($key, $value);
}
$customers_stmt->execute();
$customers = $customers_stmt->fetchAll(PDO::FETCH_ASSOC);

// Get customer statistics
$stats_query = "
    SELECT 
        COUNT(*) as total_customers,
        COUNT(CASE WHEN created_at >= DATE_SUB(NOW(), INTERVAL 30 DAY) THEN 1 END) as new_customers,
        (SELECT AVG(rating) FROM reviews) as avg_rating
    FROM users 
    WHERE role = 'customer'
";
$stats_stmt = $db->prepare($stats_query);
$stats_stmt->execute();
$stats = $stats_stmt->fetch(PDO::FETCH_ASSOC);

$page_title = 'Customer Management';
include '../includes/header.php';
?>

<div class="flex h-screen bg-gray-50">
    <?php include '../includes/sidebar.php'; ?>
    
    <div class="flex-1 flex flex-col overflow-hidden lg:ml-0">
        <?php include '../includes/dashboard_header.php'; ?>
        
        <main class="flex-1 overflow-y-auto p-6">
            <div class="space-y-6 fade-in">
                <div class="flex items-center justify-between">
                    <h1 class="text-2xl font-bold text-gray-900">Customer Management</h1>
                    <div class="text-sm text-gray-500">
                        <?php echo count($customers); ?> customer(s) found
                    </div>
                </div>

                <!-- Search and Filters -->
                <div class="bg-white rounded-lg shadow-sm border border-gray-200 p-4">
                    <form method="GET" class="flex flex-col sm:flex-row gap-4">
                        <div class="flex-1">
                            <div class="relative">
                                <i data-lucide="search" class="absolute left-3 top-3 w-4 h-4 text-gray-400"></i>
                                <input type="text" name="search" value="<?php echo htmlspecialchars($search); ?>"
                                       placeholder="Search customers..."
                                       class="w-full pl-10 pr-4 py-2 border border-gray-300 rounded-lg focus:ring-amber-500 focus:border-amber-500">
                            </div>
                        </div>
                        <div class="sm:w-48">
                            <select name="sort_by" class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-amber-500 focus:border-amber-500">
                                <option value="name" <?php echo $sort_by === 'name' ? 'selected' : ''; ?>>Sort by Name</option>
                                <option value="visits" <?php echo $sort_by === 'visits' ? 'selected' : ''; ?>>Sort by Visits</option>
                                <option value="lastVisit" <?php echo $sort_by === 'lastVisit' ? 'selected' : ''; ?>>Sort by Last Visit</option>
                            </select>
                        </div>
                        <button type="submit" class="bg-amber-600 text-white px-4 py-2 rounded-lg hover:bg-amber-700">
                            Search
                        </button>
                    </form>
                </div>

                <!-- Customer Stats -->
                <div class="grid grid-cols-1 md:grid-cols-4 gap-6">
                    <div class="bg-white rounded-lg shadow-sm border border-gray-200 p-6 hover-scale">
                        <div class="text-2xl font-bold text-blue-600"><?php echo $stats['total_customers']; ?></div>
                        <div class="text-sm text-gray-500">Total Customers</div>
                    </div>
                    <div class="bg-white rounded-lg shadow-sm border border-gray-200 p-6 hover-scale">
                        <div class="text-2xl font-bold text-green-600"><?php echo $stats['new_customers']; ?></div>
                        <div class="text-sm text-gray-500">New This Month</div>
                    </div>
                    <div class="bg-white rounded-lg shadow-sm border border-gray-200 p-6 hover-scale">
                        <div class="text-2xl font-bold text-yellow-600"><?php echo $stats['avg_rating'] ? number_format($stats['avg_rating'], 1) : 'N/A'; ?></div>
                        <div class="text-sm text-gray-500">Average Rating</div>
                    </div>
                    <div class="bg-white rounded-lg shadow-sm border border-gray-200 p-6 hover-scale">
                        <div class="text-2xl font-bold text-purple-600"><?php echo array_sum(array_column($customers, 'total_reservations')); ?></div>
                        <div class="text-sm text-gray-500">Total Reservations</div>
                    </div>
                </div>

                <!-- Customers Table -->
                <div class="bg-white rounded-lg shadow-sm border border-gray-200">
                    <div class="overflow-x-auto">
                        <table class="w-full">
                            <thead class="bg-gray-50 border-b border-gray-200">
                                <tr>
                                    <th class="text-left py-3 px-4 font-medium text-gray-900">Customer</th>
                                    <th class="text-left py-3 px-4 font-medium text-gray-900">Contact</th>
                                    <th class="text-left py-3 px-4 font-medium text-gray-900">Reservations</th>
                                    <th class="text-left py-3 px-4 font-medium text-gray-900">Last Visit</th>
                                    <th class="text-left py-3 px-4 font-medium text-gray-900">Rating</th>
                                    <th class="text-left py-3 px-4 font-medium text-gray-900">Joined</th>
                                    <th class="text-left py-3 px-4 font-medium text-gray-900">Actions</th>
                                </tr>
                            </thead>
                            <tbody class="divide-y divide-gray-200">
                                <?php foreach ($customers as $customer): ?>
                                    <tr class="hover:bg-gray-50">
                                        <td class="py-4 px-4">
                                            <div class="flex items-center">
                                                <div class="w-10 h-10 bg-amber-100 rounded-full flex items-center justify-center">
                                                    <span class="text-amber-600 font-semibold">
                                                        <?php echo strtoupper(substr($customer['name'], 0, 1)); ?>
                                                    </span>
                                                </div>
                                                <div class="ml-3">
                                                    <div class="font-medium text-gray-900"><?php echo htmlspecialchars($customer['name']); ?></div>
                                                    <div class="text-sm text-gray-500">ID: <?php echo $customer['id']; ?></div>
                                                </div>
                                            </div>
                                        </td>
                                        <td class="py-4 px-4">
                                            <div class="text-sm">
                                                <div class="flex items-center text-gray-900">
                                                    <i data-lucide="mail" class="w-3 h-3 mr-1"></i>
                                                    <?php echo htmlspecialchars($customer['email']); ?>
                                                </div>
                                                <?php if ($customer['phone']): ?>
                                                    <div class="flex items-center text-gray-500 mt-1">
                                                        <i data-lucide="phone" class="w-3 h-3 mr-1"></i>
                                                        <?php echo htmlspecialchars($customer['phone']); ?>
                                                    </div>
                                                <?php endif; ?>
                                            </div>
                                        </td>
                                        <td class="py-4 px-4">
                                            <div class="text-sm font-medium text-gray-900">
                                                <?php echo $customer['total_reservations']; ?>
                                            </div>
                                        </td>
                                        <td class="py-4 px-4">
                                            <?php if ($customer['last_visit']): ?>
                                                <div class="flex items-center text-sm text-gray-900">
                                                    <i data-lucide="calendar" class="w-4 h-4 mr-1"></i>
                                                    <?php echo date('M j, Y', strtotime($customer['last_visit'])); ?>
                                                </div>
                                            <?php else: ?>
                                                <span class="text-sm text-gray-500">Never</span>
                                            <?php endif; ?>
                                        </td>
                                        <td class="py-4 px-4">
                                            <?php if ($customer['average_rating']): ?>
                                                <div class="flex items-center">
                                                    <i data-lucide="star" class="w-4 h-4 text-yellow-400 fill-current mr-1"></i>
                                                    <span class="text-sm font-medium"><?php echo number_format($customer['average_rating'], 1); ?></span>
                                                </div>
                                            <?php else: ?>
                                                <span class="text-sm text-gray-500">No reviews</span>
                                            <?php endif; ?>
                                        </td>
                                        <td class="py-4 px-4">
                                            <div class="text-sm text-gray-900">
                                                <?php echo date('M j, Y', strtotime($customer['created_at'])); ?>
                                            </div>
                                        </td>
                                        <td class="py-4 px-4">
                                            <div class="flex space-x-2">
                                                <button onclick="viewCustomer(<?php echo $customer['id']; ?>)" class="p-1 text-gray-400 hover:text-blue-600">
                                                    <i data-lucide="eye" class="w-4 h-4"></i>
                                                </button>
                                                <button onclick="editCustomer(<?php echo $customer['id']; ?>)" class="p-1 text-gray-400 hover:text-amber-600">
                                                    <i data-lucide="edit" class="w-4 h-4"></i>
                                                </button>
                                                <button onclick="deleteCustomer(<?php echo $customer['id']; ?>)" class="p-1 text-gray-400 hover:text-red-600">
                                                    <i data-lucide="trash-2" class="w-4 h-4"></i>
                                                </button>
                                            </div>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                </div>

                <?php if (empty($customers)): ?>
                    <div class="text-center py-12">
                        <i data-lucide="users" class="w-12 h-12 text-gray-400 mx-auto mb-4"></i>
                        <h3 class="text-lg font-medium text-gray-900 mb-2">No Customers Found</h3>
                        <p class="text-gray-500">No customers match your search criteria.</p>
                    </div>
                <?php endif; ?>
            </div>
        </main>
    </div>
</div>

<script>
function viewCustomer(id) {
    alert('View customer details for ID: ' + id);
}

function editCustomer(id) {
    alert('Edit customer modal would open for ID: ' + id);
}

function deleteCustomer(id) {
    if (confirm('Are you sure you want to delete this customer?')) {
        alert('Delete customer with ID: ' + id);
    }
}
</script>

<?php include '../includes/footer.php'; ?>