<?php
require_once '../includes/auth.php';
require_once '../config/database.php';

requireAdmin();

$database = new Database();
$db = $database->getConnection();

// Get statistics
$stats_query = "
    SELECT 
        (SELECT COUNT(*) FROM reservations WHERE DATE(created_at) = CURDATE()) as today_reservations,
        (SELECT COUNT(*) FROM users WHERE role = 'customer') as total_customers,
        (SELECT COUNT(*) FROM reviews WHERE status = 'pending') as pending_reviews,
        (SELECT COUNT(*) FROM reservations WHERE status = 'confirmed' AND reservation_date >= CURDATE()) as upcoming_reservations
";
$stats_stmt = $db->prepare($stats_query);
$stats_stmt->execute();
$stats = $stats_stmt->fetch(PDO::FETCH_ASSOC);

// Get recent reservations
$recent_reservations_query = "
    SELECT r.*, u.name as customer_name, rt.table_number 
    FROM reservations r 
    JOIN users u ON r.customer_id = u.id 
    JOIN restaurant_tables rt ON r.table_id = rt.id 
    ORDER BY r.created_at DESC 
    LIMIT 5
";
$recent_stmt = $db->prepare($recent_reservations_query);
$recent_stmt->execute();
$recent_reservations = $recent_stmt->fetchAll(PDO::FETCH_ASSOC);

$page_title = 'Admin Dashboard';
include '../includes/header.php';
?>

<div class="flex h-screen bg-gray-50">
    <?php include '../includes/sidebar.php'; ?>
    
    <div class="flex-1 flex flex-col overflow-hidden lg:ml-0">
        <?php include '../includes/dashboard_header.php'; ?>
        
        <main class="flex-1 overflow-y-auto p-6">
            <div class="space-y-6 fade-in">
                <div class="flex items-center justify-between">
                    <h1 class="text-2xl font-bold text-gray-900">Dashboard Overview</h1>
                    <div class="text-sm text-gray-500">
                        Last updated: <?php echo date('M j, Y g:i A'); ?>
                    </div>
                </div>

                <!-- Stats Grid -->
                <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6">
                    <div class="bg-white rounded-lg shadow-sm border border-gray-200 p-6 hover-scale">
                        <div class="flex items-center justify-between">
                            <div>
                                <p class="text-sm font-medium text-gray-600">Today's Reservations</p>
                                <p class="text-2xl font-bold text-gray-900"><?php echo $stats['today_reservations']; ?></p>
                                <p class="text-sm text-green-600">Active today</p>
                            </div>
                            <i data-lucide="calendar" class="w-8 h-8 text-blue-600"></i>
                        </div>
                    </div>

                    <div class="bg-white rounded-lg shadow-sm border border-gray-200 p-6 hover-scale">
                        <div class="flex items-center justify-between">
                            <div>
                                <p class="text-sm font-medium text-gray-600">Total Customers</p>
                                <p class="text-2xl font-bold text-gray-900"><?php echo $stats['total_customers']; ?></p>
                                <p class="text-sm text-green-600">Registered users</p>
                            </div>
                            <i data-lucide="users" class="w-8 h-8 text-green-600"></i>
                        </div>
                    </div>

                    <div class="bg-white rounded-lg shadow-sm border border-gray-200 p-6 hover-scale">
                        <div class="flex items-center justify-between">
                            <div>
                                <p class="text-sm font-medium text-gray-600">Pending Reviews</p>
                                <p class="text-2xl font-bold text-gray-900"><?php echo $stats['pending_reviews']; ?></p>
                                <p class="text-sm text-amber-600">Need approval</p>
                            </div>
                            <i data-lucide="message-square" class="w-8 h-8 text-amber-600"></i>
                        </div>
                    </div>

                    <div class="bg-white rounded-lg shadow-sm border border-gray-200 p-6 hover-scale">
                        <div class="flex items-center justify-between">
                            <div>
                                <p class="text-sm font-medium text-gray-600">Upcoming Reservations</p>
                                <p class="text-2xl font-bold text-gray-900"><?php echo $stats['upcoming_reservations']; ?></p>
                                <p class="text-sm text-purple-600">Confirmed bookings</p>
                            </div>
                            <i data-lucide="trending-up" class="w-8 h-8 text-purple-600"></i>
                        </div>
                    </div>
                </div>

                <!-- Recent Reservations -->
                <div class="bg-white rounded-lg shadow-sm border border-gray-200 p-6">
                    <h3 class="text-lg font-semibold text-gray-900 mb-4">Recent Reservations</h3>
                    <div class="overflow-x-auto">
                        <table class="w-full">
                            <thead class="bg-gray-50 border-b border-gray-200">
                                <tr>
                                    <th class="text-left py-3 px-4 font-medium text-gray-900">Customer</th>
                                    <th class="text-left py-3 px-4 font-medium text-gray-900">Table</th>
                                    <th class="text-left py-3 px-4 font-medium text-gray-900">Date & Time</th>
                                    <th class="text-left py-3 px-4 font-medium text-gray-900">Guests</th>
                                    <th class="text-left py-3 px-4 font-medium text-gray-900">Status</th>
                                </tr>
                            </thead>
                            <tbody class="divide-y divide-gray-200">
                                <?php foreach ($recent_reservations as $reservation): ?>
                                    <tr class="hover:bg-gray-50">
                                        <td class="py-4 px-4">
                                            <div class="font-medium text-gray-900"><?php echo htmlspecialchars($reservation['customer_name']); ?></div>
                                        </td>
                                        <td class="py-4 px-4">
                                            <div class="font-medium">Table <?php echo $reservation['table_number']; ?></div>
                                        </td>
                                        <td class="py-4 px-4">
                                            <div class="text-sm">
                                                <div class="font-medium"><?php echo date('M j, Y', strtotime($reservation['reservation_date'])); ?></div>
                                                <div class="text-gray-500"><?php echo date('g:i A', strtotime($reservation['reservation_time'])); ?></div>
                                            </div>
                                        </td>
                                        <td class="py-4 px-4">
                                            <div class="text-sm"><?php echo $reservation['guests']; ?> guests</div>
                                        </td>
                                        <td class="py-4 px-4">
                                            <?php
                                            $status_colors = [
                                                'pending' => 'bg-yellow-100 text-yellow-800',
                                                'confirmed' => 'bg-green-100 text-green-800',
                                                'completed' => 'bg-blue-100 text-blue-800',
                                                'cancelled' => 'bg-red-100 text-red-800'
                                            ];
                                            $color_class = $status_colors[$reservation['status']] ?? 'bg-gray-100 text-gray-800';
                                            ?>
                                            <span class="inline-flex px-2 py-1 text-xs font-semibold rounded-full <?php echo $color_class; ?>">
                                                <?php echo ucfirst($reservation['status']); ?>
                                            </span>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                </div>

                <!-- Quick Actions -->
                <div class="grid grid-cols-1 md:grid-cols-3 gap-6">
                    <a href="/admin/reservations.php" class="bg-white rounded-lg shadow-sm border border-gray-200 p-6 hover:shadow-md transition-shadow hover-scale">
                        <div class="flex items-center">
                            <i data-lucide="calendar-plus" class="w-8 h-8 text-blue-600 mr-4"></i>
                            <div>
                                <h3 class="font-medium text-gray-900">Manage Reservations</h3>
                                <p class="text-sm text-gray-500">View and manage all bookings</p>
                            </div>
                        </div>
                    </a>

                    <a href="/admin/tables.php" class="bg-white rounded-lg shadow-sm border border-gray-200 p-6 hover:shadow-md transition-shadow hover-scale">
                        <div class="flex items-center">
                            <i data-lucide="table" class="w-8 h-8 text-green-600 mr-4"></i>
                            <div>
                                <h3 class="font-medium text-gray-900">Table Management</h3>
                                <p class="text-sm text-gray-500">Update table status and availability</p>
                            </div>
                        </div>
                    </a>

                    <a href="/admin/analytics.php" class="bg-white rounded-lg shadow-sm border border-gray-200 p-6 hover:shadow-md transition-shadow hover-scale">
                        <div class="flex items-center">
                            <i data-lucide="bar-chart-3" class="w-8 h-8 text-purple-600 mr-4"></i>
                            <div>
                                <h3 class="font-medium text-gray-900">View Analytics</h3>
                                <p class="text-sm text-gray-500">Check performance metrics</p>
                            </div>
                        </div>
                    </a>
                </div>
            </div>
        </main>
    </div>
</div>

<?php include '../includes/footer.php'; ?>