<?php
require_once '../includes/auth.php';
require_once '../config/database.php';

requireAdmin();

$database = new Database();
$db = $database->getConnection();

$success = '';
$error = '';

// Handle sending new notification
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action']) && $_POST['action'] === 'send_notification') {
    $title = trim($_POST['title']);
    $message = trim($_POST['message']);
    $type = $_POST['type'];
    $recipient_type = $_POST['recipient_type'];
    
    if (empty($title) || empty($message)) {
        $error = 'Please fill in all required fields.';
    } else {
        // Get recipients based on type
        if ($recipient_type === 'all') {
            $recipients_query = "SELECT id FROM users WHERE role = 'customer'";
        } else {
            $recipients_query = "SELECT DISTINCT customer_id as id FROM reservations WHERE status = 'confirmed' AND reservation_date >= CURDATE()";
        }
        
        $recipients_stmt = $db->prepare($recipients_query);
        $recipients_stmt->execute();
        $recipients = $recipients_stmt->fetchAll(PDO::FETCH_ASSOC);
        
        // Insert notification for each recipient
        $insert_query = "INSERT INTO notifications (user_id, title, message, type) VALUES (:user_id, :title, :message, :type)";
        $insert_stmt = $db->prepare($insert_query);
        
        $sent_count = 0;
        foreach ($recipients as $recipient) {
            $insert_stmt->bindParam(':user_id', $recipient['id']);
            $insert_stmt->bindParam(':title', $title);
            $insert_stmt->bindParam(':message', $message);
            $insert_stmt->bindParam(':type', $type);
            
            if ($insert_stmt->execute()) {
                $sent_count++;
            }
        }
        
        if ($sent_count > 0) {
            $success = "Notification sent to $sent_count customer(s) successfully!";
        } else {
            $error = 'Failed to send notifications.';
        }
    }
}

// Get filter parameters
$filter_type = $_GET['type'] ?? 'all';

// Build query with filters
$where_clause = '';
$params = [];

if ($filter_type !== 'all') {
    $where_clause = 'WHERE n.type = :type';
    $params[':type'] = $filter_type;
}

// Get notifications with user info
$notifications_query = "
    SELECT n.*, u.name as user_name, u.email as user_email
    FROM notifications n
    JOIN users u ON n.user_id = u.id
    $where_clause
    ORDER BY n.created_at DESC
    LIMIT 50
";

$notifications_stmt = $db->prepare($notifications_query);
foreach ($params as $key => $value) {
    $notifications_stmt->bindValue($key, $value);
}
$notifications_stmt->execute();
$notifications = $notifications_stmt->fetchAll(PDO::FETCH_ASSOC);

// Get notification statistics
$stats_query = "
    SELECT 
        COUNT(*) as total_sent,
        COUNT(CASE WHEN is_read = 1 THEN 1 END) as delivered,
        COUNT(CASE WHEN is_read = 0 THEN 1 END) as pending,
        COUNT(CASE WHEN created_at >= DATE_SUB(NOW(), INTERVAL 24 HOUR) THEN 1 END) as last_24h
    FROM notifications
";
$stats_stmt = $db->prepare($stats_query);
$stats_stmt->execute();
$stats = $stats_stmt->fetch(PDO::FETCH_ASSOC);

$page_title = 'Notifications Management';
include '../includes/header.php';
?>

<div class="flex h-screen bg-gray-50">
    <?php include '../includes/sidebar.php'; ?>
    
    <div class="flex-1 flex flex-col overflow-hidden lg:ml-0">
        <?php include '../includes/dashboard_header.php'; ?>
        
        <main class="flex-1 overflow-y-auto p-6">
            <div class="space-y-6 fade-in">
                <div class="flex items-center justify-between">
                    <h1 class="text-2xl font-bold text-gray-900">Notifications</h1>
                    <button onclick="showNotificationModal()" class="bg-amber-600 text-white px-4 py-2 rounded-lg hover:bg-amber-700 flex items-center hover-scale">
                        <i data-lucide="send" class="w-4 h-4 mr-2"></i>
                        Send Notification
                    </button>
                </div>

                <?php if ($success): ?>
                    <div class="bg-green-50 border border-green-200 text-green-600 px-4 py-3 rounded-lg alert-auto-hide">
                        <?php echo htmlspecialchars($success); ?>
                    </div>
                <?php endif; ?>

                <?php if ($error): ?>
                    <div class="bg-red-50 border border-red-200 text-red-600 px-4 py-3 rounded-lg alert-auto-hide">
                        <?php echo htmlspecialchars($error); ?>
                    </div>
                <?php endif; ?>

                <!-- Quick Stats -->
                <div class="grid grid-cols-1 md:grid-cols-4 gap-6">
                    <div class="bg-white rounded-lg shadow-sm border border-gray-200 p-6 hover-scale">
                        <div class="flex items-center justify-between">
                            <div>
                                <div class="text-2xl font-bold text-blue-600"><?php echo $stats['total_sent']; ?></div>
                                <div class="text-sm text-gray-500">Total Sent</div>
                            </div>
                            <i data-lucide="send" class="w-8 h-8 text-blue-600"></i>
                        </div>
                    </div>
                    <div class="bg-white rounded-lg shadow-sm border border-gray-200 p-6 hover-scale">
                        <div class="flex items-center justify-between">
                            <div>
                                <div class="text-2xl font-bold text-green-600"><?php echo $stats['delivered']; ?></div>
                                <div class="text-sm text-gray-500">Read</div>
                            </div>
                            <i data-lucide="check" class="w-8 h-8 text-green-600"></i>
                        </div>
                    </div>
                    <div class="bg-white rounded-lg shadow-sm border border-gray-200 p-6 hover-scale">
                        <div class="flex items-center justify-between">
                            <div>
                                <div class="text-2xl font-bold text-yellow-600"><?php echo $stats['pending']; ?></div>
                                <div class="text-sm text-gray-500">Unread</div>
                            </div>
                            <i data-lucide="clock" class="w-8 h-8 text-yellow-600"></i>
                        </div>
                    </div>
                    <div class="bg-white rounded-lg shadow-sm border border-gray-200 p-6 hover-scale">
                        <div class="flex items-center justify-between">
                            <div>
                                <div class="text-2xl font-bold text-purple-600"><?php echo $stats['last_24h']; ?></div>
                                <div class="text-sm text-gray-500">Last 24h</div>
                            </div>
                            <i data-lucide="bell" class="w-8 h-8 text-purple-600"></i>
                        </div>
                    </div>
                </div>

                <!-- Filters -->
                <div class="bg-white rounded-lg shadow-sm border border-gray-200 p-4">
                    <form method="GET" class="flex space-x-4">
                        <div class="flex items-center space-x-2">
                            <i data-lucide="filter" class="w-4 h-4 text-gray-400"></i>
                            <span class="text-sm font-medium text-gray-700">Filter by:</span>
                        </div>
                        
                        <div class="flex flex-wrap gap-2">
                            <select name="type" class="px-3 py-1 border border-gray-300 rounded-lg text-sm focus:ring-amber-500 focus:border-amber-500">
                                <option value="all" <?php echo $filter_type === 'all' ? 'selected' : ''; ?>>All Types</option>
                                <option value="booking" <?php echo $filter_type === 'booking' ? 'selected' : ''; ?>>Booking Updates</option>
                                <option value="reminder" <?php echo $filter_type === 'reminder' ? 'selected' : ''; ?>>Reminders</option>
                                <option value="review" <?php echo $filter_type === 'review' ? 'selected' : ''; ?>>Reviews</option>
                                <option value="general" <?php echo $filter_type === 'general' ? 'selected' : ''; ?>>General</option>
                            </select>
                            <button type="submit" class="bg-amber-600 text-white px-3 py-1 rounded-lg hover:bg-amber-700 text-sm">
                                Apply
                            </button>
                        </div>

                        <?php if ($filter_type !== 'all'): ?>
                            <a href="?type=all" class="px-3 py-1 text-sm text-gray-600 hover:text-gray-800">
                                Clear filters
                            </a>
                        <?php endif; ?>
                    </form>
                </div>

                <!-- Notifications List -->
                <div class="space-y-3">
                    <?php foreach ($notifications as $notification): ?>
                        <?php
                        $type_colors = [
                            'booking' => 'text-blue-600',
                            'review' => 'text-yellow-600',
                            'reminder' => 'text-purple-600',
                            'general' => 'text-gray-600'
                        ];
                        $type_icons = [
                            'booking' => 'calendar',
                            'review' => 'star',
                            'reminder' => 'bell',
                            'general' => 'message-square'
                        ];
                        $icon_color = $type_colors[$notification['type']] ?? 'text-gray-600';
                        $icon_name = $type_icons[$notification['type']] ?? 'message-square';
                        ?>
                        <div class="bg-white rounded-lg shadow-sm border p-4 transition-all hover:shadow-md hover-scale <?php echo $notification['is_read'] ? 'border-gray-200' : 'border-amber-200 bg-amber-50'; ?>">
                            <div class="flex items-start space-x-3">
                                <div class="p-2 rounded-lg <?php echo $notification['is_read'] ? 'bg-gray-100' : 'bg-amber-100'; ?>">
                                    <i data-lucide="<?php echo $icon_name; ?>" class="w-5 h-5 <?php echo $icon_color; ?>"></i>
                                </div>
                                
                                <div class="flex-1 min-w-0">
                                    <div class="flex items-center justify-between mb-1">
                                        <h3 class="font-medium <?php echo $notification['is_read'] ? 'text-gray-700' : 'text-gray-900'; ?>">
                                            <?php echo htmlspecialchars($notification['title']); ?>
                                        </h3>
                                        <div class="flex items-center space-x-2">
                                            <span class="text-sm text-gray-500">
                                                <?php echo date('M j, Y g:i A', strtotime($notification['created_at'])); ?>
                                            </span>
                                            <?php if (!$notification['is_read']): ?>
                                                <span class="w-2 h-2 bg-amber-500 rounded-full"></span>
                                            <?php endif; ?>
                                        </div>
                                    </div>
                                    
                                    <p class="text-sm mb-2 <?php echo $notification['is_read'] ? 'text-gray-500' : 'text-gray-700'; ?>">
                                        <?php echo htmlspecialchars($notification['message']); ?>
                                    </p>
                                    
                                    <div class="flex items-center justify-between">
                                        <div class="flex items-center space-x-4">
                                            <span class="text-xs text-gray-500">
                                                To: <?php echo htmlspecialchars($notification['user_name']); ?>
                                            </span>
                                            <?php
                                            $type_badge_colors = [
                                                'booking' => 'bg-blue-100 text-blue-800',
                                                'review' => 'bg-yellow-100 text-yellow-800',
                                                'reminder' => 'bg-purple-100 text-purple-800',
                                                'general' => 'bg-gray-100 text-gray-800'
                                            ];
                                            $badge_color = $type_badge_colors[$notification['type']] ?? 'bg-gray-100 text-gray-800';
                                            ?>
                                            <span class="inline-flex px-2 py-1 text-xs font-semibold rounded-full <?php echo $badge_color; ?>">
                                                <?php echo ucfirst($notification['type']); ?>
                                            </span>
                                        </div>
                                        
                                        <div class="text-xs text-gray-500">
                                            <?php echo $notification['is_read'] ? 'Read' : 'Unread'; ?>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                    <?php endforeach; ?>
                </div>

                <?php if (empty($notifications)): ?>
                    <div class="text-center py-12">
                        <i data-lucide="bell" class="w-12 h-12 text-gray-400 mx-auto mb-4"></i>
                        <h3 class="text-lg font-medium text-gray-900 mb-2">No Notifications Found</h3>
                        <p class="text-gray-500">No notifications match your current filters.</p>
                    </div>
                <?php endif; ?>
            </div>
        </main>
    </div>
</div>

<!-- Notification Modal -->
<div id="notificationModal" class="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50 hidden">
    <div class="bg-white rounded-lg p-6 w-full max-w-md mx-4">
        <h3 class="text-lg font-semibold text-gray-900 mb-4">Send New Notification</h3>
        <form method="POST">
            <input type="hidden" name="action" value="send_notification">
            
            <div class="space-y-4">
                <div>
                    <label class="block text-sm font-medium text-gray-700 mb-1">Type</label>
                    <select name="type" required class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-amber-500 focus:border-amber-500">
                        <option value="general">General</option>
                        <option value="booking">Booking</option>
                        <option value="reminder">Reminder</option>
                        <option value="review">Review</option>
                    </select>
                </div>
                
                <div>
                    <label class="block text-sm font-medium text-gray-700 mb-1">Recipients</label>
                    <select name="recipient_type" required class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-amber-500 focus:border-amber-500">
                        <option value="all">All Customers</option>
                        <option value="active">Customers with Upcoming Reservations</option>
                    </select>
                </div>
                
                <div>
                    <label class="block text-sm font-medium text-gray-700 mb-1">Title</label>
                    <input type="text" name="title" required
                           class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-amber-500 focus:border-amber-500"
                           placeholder="Notification title">
                </div>
                
                <div>
                    <label class="block text-sm font-medium text-gray-700 mb-1">Message</label>
                    <textarea name="message" rows="3" required
                              class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-amber-500 focus:border-amber-500"
                              placeholder="Notification message"></textarea>
                </div>
            </div>
            
            <div class="flex space-x-3 mt-6">
                <button type="button" onclick="hideNotificationModal()"
                        class="flex-1 px-4 py-2 border border-gray-300 text-gray-700 rounded-lg hover:bg-gray-50">
                    Cancel
                </button>
                <button type="submit"
                        class="flex-1 px-4 py-2 bg-amber-600 text-white rounded-lg hover:bg-amber-700">
                    Send
                </button>
            </div>
        </form>
    </div>
</div>

<script>
function showNotificationModal() {
    document.getElementById('notificationModal').classList.remove('hidden');
}

function hideNotificationModal() {
    document.getElementById('notificationModal').classList.add('hidden');
}
</script>

<?php include '../includes/footer.php'; ?>