<?php
require_once '../includes/auth.php';
require_once '../config/database.php';

requireAdmin();

$database = new Database();
$db = $database->getConnection();

$success = '';
$error = '';

// Handle reservation status updates
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action'])) {
    $reservation_id = $_POST['reservation_id'];
    $action = $_POST['action'];
    
    $new_status = '';
    switch ($action) {
        case 'confirm':
            $new_status = 'confirmed';
            break;
        case 'complete':
            $new_status = 'completed';
            break;
        case 'cancel':
            $new_status = 'cancelled';
            break;
    }
    
    if ($new_status) {
        $update_query = "UPDATE reservations SET status = :status WHERE id = :id";
        $update_stmt = $db->prepare($update_query);
        $update_stmt->bindParam(':status', $new_status);
        $update_stmt->bindParam(':id', $reservation_id);
        
        if ($update_stmt->execute()) {
            $success = 'Reservation status updated successfully!';
        } else {
            $error = 'Failed to update reservation status.';
        }
    }
}

// Get filter parameters
$status_filter = $_GET['status'] ?? 'all';
$search = $_GET['search'] ?? '';

// Build query with filters
$where_conditions = [];
$params = [];

if ($status_filter !== 'all') {
    $where_conditions[] = "r.status = :status";
    $params[':status'] = $status_filter;
}

if (!empty($search)) {
    $where_conditions[] = "(u.name LIKE :search OR u.email LIKE :search)";
    $params[':search'] = '%' . $search . '%';
}

$where_clause = !empty($where_conditions) ? 'WHERE ' . implode(' AND ', $where_conditions) : '';

// Get reservations with customer and table info
$reservations_query = "
    SELECT r.*, u.name as customer_name, u.email as customer_email, u.phone as customer_phone,
           rt.table_number, rt.location
    FROM reservations r
    JOIN users u ON r.customer_id = u.id
    JOIN restaurant_tables rt ON r.table_id = rt.id
    $where_clause
    ORDER BY r.created_at DESC
";

$reservations_stmt = $db->prepare($reservations_query);
foreach ($params as $key => $value) {
    $reservations_stmt->bindValue($key, $value);
}
$reservations_stmt->execute();
$reservations = $reservations_stmt->fetchAll(PDO::FETCH_ASSOC);

$page_title = 'Reservations Management';
include '../includes/header.php';
?>

<div class="flex h-screen bg-gray-50">
    <?php include '../includes/sidebar.php'; ?>
    
    <div class="flex-1 flex flex-col overflow-hidden lg:ml-0">
        <?php include '../includes/dashboard_header.php'; ?>
        
        <main class="flex-1 overflow-y-auto p-6">
            <div class="space-y-6 fade-in">
                <div class="flex items-center justify-between">
                    <h1 class="text-2xl font-bold text-gray-900">Reservations Management</h1>
                    <button onclick="showNewReservationModal()" class="bg-amber-600 text-white px-4 py-2 rounded-lg hover:bg-amber-700 flex items-center hover-scale">
                        <i data-lucide="plus" class="w-4 h-4 mr-2"></i>
                        New Reservation
                    </button>
                </div>

                <?php if ($success): ?>
                    <div class="bg-green-50 border border-green-200 text-green-600 px-4 py-3 rounded-lg alert-auto-hide">
                        <?php echo htmlspecialchars($success); ?>
                    </div>
                <?php endif; ?>

                <?php if ($error): ?>
                    <div class="bg-red-50 border border-red-200 text-red-600 px-4 py-3 rounded-lg alert-auto-hide">
                        <?php echo htmlspecialchars($error); ?>
                    </div>
                <?php endif; ?>

                <!-- Filters -->
                <div class="bg-white rounded-lg shadow-sm border border-gray-200 p-4">
                    <form method="GET" class="flex flex-col sm:flex-row gap-4">
                        <div class="flex-1">
                            <div class="relative">
                                <i data-lucide="search" class="absolute left-3 top-3 w-4 h-4 text-gray-400"></i>
                                <input type="text" name="search" value="<?php echo htmlspecialchars($search); ?>"
                                       placeholder="Search by customer name or email..."
                                       class="w-full pl-10 pr-4 py-2 border border-gray-300 rounded-lg focus:ring-amber-500 focus:border-amber-500">
                            </div>
                        </div>
                        <div class="sm:w-48">
                            <select name="status" class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-amber-500 focus:border-amber-500">
                                <option value="all" <?php echo $status_filter === 'all' ? 'selected' : ''; ?>>All Status</option>
                                <option value="pending" <?php echo $status_filter === 'pending' ? 'selected' : ''; ?>>Pending</option>
                                <option value="confirmed" <?php echo $status_filter === 'confirmed' ? 'selected' : ''; ?>>Confirmed</option>
                                <option value="completed" <?php echo $status_filter === 'completed' ? 'selected' : ''; ?>>Completed</option>
                                <option value="cancelled" <?php echo $status_filter === 'cancelled' ? 'selected' : ''; ?>>Cancelled</option>
                            </select>
                        </div>
                        <button type="submit" class="bg-amber-600 text-white px-4 py-2 rounded-lg hover:bg-amber-700">
                            Filter
                        </button>
                    </form>
                </div>

                <!-- Reservations Table -->
                <div class="bg-white rounded-lg shadow-sm border border-gray-200">
                    <div class="overflow-x-auto">
                        <table class="w-full">
                            <thead class="bg-gray-50 border-b border-gray-200">
                                <tr>
                                    <th class="text-left py-3 px-4 font-medium text-gray-900">Customer</th>
                                    <th class="text-left py-3 px-4 font-medium text-gray-900">Date & Time</th>
                                    <th class="text-left py-3 px-4 font-medium text-gray-900">Table</th>
                                    <th class="text-left py-3 px-4 font-medium text-gray-900">Guests</th>
                                    <th class="text-left py-3 px-4 font-medium text-gray-900">Status</th>
                                    <th class="text-left py-3 px-4 font-medium text-gray-900">Actions</th>
                                </tr>
                            </thead>
                            <tbody class="divide-y divide-gray-200">
                                <?php foreach ($reservations as $reservation): ?>
                                    <tr class="hover:bg-gray-50">
                                        <td class="py-4 px-4">
                                            <div>
                                                <div class="font-medium text-gray-900"><?php echo htmlspecialchars($reservation['customer_name']); ?></div>
                                                <div class="text-sm text-gray-500 flex items-center">
                                                    <i data-lucide="mail" class="w-3 h-3 mr-1"></i>
                                                    <?php echo htmlspecialchars($reservation['customer_email']); ?>
                                                </div>
                                                <?php if ($reservation['customer_phone']): ?>
                                                    <div class="text-sm text-gray-500 flex items-center">
                                                        <i data-lucide="phone" class="w-3 h-3 mr-1"></i>
                                                        <?php echo htmlspecialchars($reservation['customer_phone']); ?>
                                                    </div>
                                                <?php endif; ?>
                                            </div>
                                        </td>
                                        <td class="py-4 px-4">
                                            <div class="flex items-center text-sm">
                                                <i data-lucide="calendar" class="w-4 h-4 mr-2 text-gray-400"></i>
                                                <div>
                                                    <div class="font-medium"><?php echo date('M j, Y', strtotime($reservation['reservation_date'])); ?></div>
                                                    <div class="text-gray-500 flex items-center">
                                                        <i data-lucide="clock" class="w-3 h-3 mr-1"></i>
                                                        <?php echo date('g:i A', strtotime($reservation['reservation_time'])); ?>
                                                    </div>
                                                </div>
                                            </div>
                                        </td>
                                        <td class="py-4 px-4">
                                            <div class="font-medium">Table <?php echo $reservation['table_number']; ?></div>
                                            <div class="text-sm text-gray-500"><?php echo htmlspecialchars($reservation['location']); ?></div>
                                        </td>
                                        <td class="py-4 px-4">
                                            <div class="flex items-center text-sm">
                                                <i data-lucide="users" class="w-4 h-4 mr-1 text-gray-400"></i>
                                                <?php echo $reservation['guests']; ?>
                                            </div>
                                        </td>
                                        <td class="py-4 px-4">
                                            <?php
                                            $status_colors = [
                                                'pending' => 'bg-yellow-100 text-yellow-800',
                                                'confirmed' => 'bg-green-100 text-green-800',
                                                'completed' => 'bg-blue-100 text-blue-800',
                                                'cancelled' => 'bg-red-100 text-red-800'
                                            ];
                                            $color_class = $status_colors[$reservation['status']] ?? 'bg-gray-100 text-gray-800';
                                            ?>
                                            <span class="inline-flex px-2 py-1 text-xs font-semibold rounded-full <?php echo $color_class; ?>">
                                                <?php echo ucfirst($reservation['status']); ?>
                                            </span>
                                        </td>
                                        <td class="py-4 px-4">
                                            <div class="flex space-x-2">
                                                <?php if ($reservation['status'] === 'pending'): ?>
                                                    <form method="POST" class="inline">
                                                        <input type="hidden" name="reservation_id" value="<?php echo $reservation['id']; ?>">
                                                        <input type="hidden" name="action" value="confirm">
                                                        <button type="submit" class="text-green-600 hover:text-green-700 text-sm font-medium">
                                                            Confirm
                                                        </button>
                                                    </form>
                                                    <form method="POST" class="inline">
                                                        <input type="hidden" name="reservation_id" value="<?php echo $reservation['id']; ?>">
                                                        <input type="hidden" name="action" value="cancel">
                                                        <button type="submit" class="text-red-600 hover:text-red-700 text-sm font-medium">
                                                            Cancel
                                                        </button>
                                                    </form>
                                                <?php elseif ($reservation['status'] === 'confirmed'): ?>
                                                    <form method="POST" class="inline">
                                                        <input type="hidden" name="reservation_id" value="<?php echo $reservation['id']; ?>">
                                                        <input type="hidden" name="action" value="complete">
                                                        <button type="submit" class="text-blue-600 hover:text-blue-700 text-sm font-medium">
                                                            Complete
                                                        </button>
                                                    </form>
                                                <?php endif; ?>
                                                <button onclick="viewReservation(<?php echo $reservation['id']; ?>)" class="text-amber-600 hover:text-amber-700 text-sm font-medium">
                                                    View
                                                </button>
                                            </div>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>
        </main>
    </div>
</div>

<script>
function showNewReservationModal() {
    alert('New reservation modal would open here');
}

function viewReservation(id) {
    alert('View reservation details for ID: ' + id);
}
</script>

<?php include '../includes/footer.php'; ?>