<?php
require_once '../includes/auth.php';
require_once '../config/database.php';

requireAdmin();

$database = new Database();
$db = $database->getConnection();

$success = '';
$error = '';

// Handle review status updates
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action'])) {
    $review_id = $_POST['review_id'];
    $action = $_POST['action'];
    
    $new_status = '';
    switch ($action) {
        case 'approve':
            $new_status = 'approved';
            break;
        case 'reject':
            $new_status = 'rejected';
            break;
    }
    
    if ($new_status) {
        $update_query = "UPDATE reviews SET status = :status WHERE id = :id";
        $update_stmt = $db->prepare($update_query);
        $update_stmt->bindParam(':status', $new_status);
        $update_stmt->bindParam(':id', $review_id);
        
        if ($update_stmt->execute()) {
            $success = 'Review status updated successfully!';
        } else {
            $error = 'Failed to update review status.';
        }
    }
}

// Get filter parameters
$status_filter = $_GET['status'] ?? 'all';
$rating_filter = $_GET['rating'] ?? 'all';
$search = $_GET['search'] ?? '';

// Build query with filters
$where_conditions = [];
$params = [];

if ($status_filter !== 'all') {
    $where_conditions[] = "r.status = :status";
    $params[':status'] = $status_filter;
}

if ($rating_filter !== 'all') {
    $where_conditions[] = "r.rating = :rating";
    $params[':rating'] = $rating_filter;
}

if (!empty($search)) {
    $where_conditions[] = "(u.name LIKE :search OR r.comment LIKE :search)";
    $params[':search'] = '%' . $search . '%';
}

$where_clause = !empty($where_conditions) ? 'WHERE ' . implode(' AND ', $where_conditions) : '';

// Get reviews with customer info
$reviews_query = "
    SELECT r.*, u.name as customer_name, u.email as customer_email
    FROM reviews r
    JOIN users u ON r.customer_id = u.id
    $where_clause
    ORDER BY r.created_at DESC
";

$reviews_stmt = $db->prepare($reviews_query);
foreach ($params as $key => $value) {
    $reviews_stmt->bindValue($key, $value);
}
$reviews_stmt->execute();
$reviews = $reviews_stmt->fetchAll(PDO::FETCH_ASSOC);

$page_title = 'Reviews Management';
include '../includes/header.php';
?>

<div class="flex h-screen bg-gray-50">
    <?php include '../includes/sidebar.php'; ?>
    
    <div class="flex-1 flex flex-col overflow-hidden lg:ml-0">
        <?php include '../includes/dashboard_header.php'; ?>
        
        <main class="flex-1 overflow-y-auto p-6">
            <div class="space-y-6 fade-in">
                <div class="flex items-center justify-between">
                    <h1 class="text-2xl font-bold text-gray-900">Customer Reviews</h1>
                    <div class="text-sm text-gray-500">
                        <?php echo count($reviews); ?> review(s) found
                    </div>
                </div>

                <?php if ($success): ?>
                    <div class="bg-green-50 border border-green-200 text-green-600 px-4 py-3 rounded-lg alert-auto-hide">
                        <?php echo htmlspecialchars($success); ?>
                    </div>
                <?php endif; ?>

                <?php if ($error): ?>
                    <div class="bg-red-50 border border-red-200 text-red-600 px-4 py-3 rounded-lg alert-auto-hide">
                        <?php echo htmlspecialchars($error); ?>
                    </div>
                <?php endif; ?>

                <!-- Filters -->
                <div class="bg-white rounded-lg shadow-sm border border-gray-200 p-4">
                    <form method="GET" class="flex flex-col lg:flex-row gap-4">
                        <div class="flex-1">
                            <div class="relative">
                                <i data-lucide="search" class="absolute left-3 top-3 w-4 h-4 text-gray-400"></i>
                                <input type="text" name="search" value="<?php echo htmlspecialchars($search); ?>"
                                       placeholder="Search reviews..."
                                       class="w-full pl-10 pr-4 py-2 border border-gray-300 rounded-lg focus:ring-amber-500 focus:border-amber-500">
                            </div>
                        </div>
                        <div class="w-48">
                            <select name="status" class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-amber-500 focus:border-amber-500">
                                <option value="all" <?php echo $status_filter === 'all' ? 'selected' : ''; ?>>All Status</option>
                                <option value="pending" <?php echo $status_filter === 'pending' ? 'selected' : ''; ?>>Pending</option>
                                <option value="approved" <?php echo $status_filter === 'approved' ? 'selected' : ''; ?>>Approved</option>
                                <option value="rejected" <?php echo $status_filter === 'rejected' ? 'selected' : ''; ?>>Rejected</option>
                            </select>
                        </div>
                        <div class="w-48">
                            <select name="rating" class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-amber-500 focus:border-amber-500">
                                <option value="all" <?php echo $rating_filter === 'all' ? 'selected' : ''; ?>>All Ratings</option>
                                <option value="5" <?php echo $rating_filter === '5' ? 'selected' : ''; ?>>5 Stars</option>
                                <option value="4" <?php echo $rating_filter === '4' ? 'selected' : ''; ?>>4 Stars</option>
                                <option value="3" <?php echo $rating_filter === '3' ? 'selected' : ''; ?>>3 Stars</option>
                                <option value="2" <?php echo $rating_filter === '2' ? 'selected' : ''; ?>>2 Stars</option>
                                <option value="1" <?php echo $rating_filter === '1' ? 'selected' : ''; ?>>1 Star</option>
                            </select>
                        </div>
                        <button type="submit" class="bg-amber-600 text-white px-4 py-2 rounded-lg hover:bg-amber-700">
                            Filter
                        </button>
                    </form>
                </div>

                <!-- Reviews List -->
                <div class="space-y-4">
                    <?php foreach ($reviews as $review): ?>
                        <div class="bg-white rounded-lg shadow-sm border border-gray-200 p-6 hover-scale">
                            <div class="flex items-start justify-between">
                                <div class="flex-1">
                                    <div class="flex items-center justify-between mb-3">
                                        <div class="flex items-center space-x-3">
                                            <div class="w-10 h-10 bg-amber-100 rounded-full flex items-center justify-center">
                                                <span class="text-amber-600 font-semibold">
                                                    <?php echo strtoupper(substr($review['customer_name'], 0, 1)); ?>
                                                </span>
                                            </div>
                                            <div>
                                                <h3 class="font-medium text-gray-900"><?php echo htmlspecialchars($review['customer_name']); ?></h3>
                                                <p class="text-sm text-gray-500"><?php echo date('M j, Y', strtotime($review['created_at'])); ?></p>
                                            </div>
                                        </div>
                                        <div class="flex items-center space-x-3">
                                            <div class="flex">
                                                <?php for ($i = 1; $i <= 5; $i++): ?>
                                                    <i data-lucide="star" class="w-4 h-4 <?php echo $i <= $review['rating'] ? 'text-yellow-400 fill-current' : 'text-gray-300'; ?>"></i>
                                                <?php endfor; ?>
                                            </div>
                                            <?php
                                            $status_colors = [
                                                'pending' => 'bg-yellow-100 text-yellow-800',
                                                'approved' => 'bg-green-100 text-green-800',
                                                'rejected' => 'bg-red-100 text-red-800'
                                            ];
                                            $color_class = $status_colors[$review['status']] ?? 'bg-gray-100 text-gray-800';
                                            ?>
                                            <span class="inline-flex px-2 py-1 text-xs font-semibold rounded-full <?php echo $color_class; ?>">
                                                <?php echo ucfirst($review['status']); ?>
                                            </span>
                                        </div>
                                    </div>
                                    
                                    <p class="text-gray-700 mb-4"><?php echo htmlspecialchars($review['comment']); ?></p>
                                    
                                    <?php if ($review['reservation_id']): ?>
                                        <p class="text-sm text-gray-500 mb-4">
                                            Related to reservation #<?php echo $review['reservation_id']; ?>
                                        </p>
                                    <?php endif; ?>
                                    
                                    <?php if ($review['status'] === 'pending'): ?>
                                        <div class="flex space-x-2">
                                            <form method="POST" class="inline">
                                                <input type="hidden" name="review_id" value="<?php echo $review['id']; ?>">
                                                <input type="hidden" name="action" value="approve">
                                                <button type="submit" class="flex items-center px-3 py-1 text-sm text-green-600 hover:text-green-700 hover:bg-green-50 rounded-lg transition-colors">
                                                    <i data-lucide="thumbs-up" class="w-4 h-4 mr-1"></i>
                                                    Approve
                                                </button>
                                            </form>
                                            <form method="POST" class="inline">
                                                <input type="hidden" name="review_id" value="<?php echo $review['id']; ?>">
                                                <input type="hidden" name="action" value="reject">
                                                <button type="submit" class="flex items-center px-3 py-1 text-sm text-red-600 hover:text-red-700 hover:bg-red-50 rounded-lg transition-colors">
                                                    <i data-lucide="thumbs-down" class="w-4 h-4 mr-1"></i>
                                                    Reject
                                                </button>
                                            </form>
                                        </div>
                                    <?php endif; ?>
                                </div>
                            </div>
                        </div>
                    <?php endforeach; ?>
                </div>

                <?php if (empty($reviews)): ?>
                    <div class="text-center py-12">
                        <i data-lucide="message-square" class="w-12 h-12 text-gray-400 mx-auto mb-4"></i>
                        <h3 class="text-lg font-medium text-gray-900 mb-2">No Reviews Found</h3>
                        <p class="text-gray-500">No reviews match your current filters.</p>
                    </div>
                <?php endif; ?>
            </div>
        </main>
    </div>
</div>

<?php include '../includes/footer.php'; ?>