<?php
require_once '../includes/auth.php';
require_once '../config/database.php';

requireAdmin();

$database = new Database();
$db = $database->getConnection();

$success = '';
$error = '';

// Handle settings update
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $settings = [
        'restaurant_name' => $_POST['restaurant_name'] ?? '',
        'restaurant_location' => $_POST['restaurant_location'] ?? '',
        'restaurant_phone' => $_POST['restaurant_phone'] ?? '',
        'restaurant_email' => $_POST['restaurant_email'] ?? '',
        'advance_booking_days' => $_POST['advance_booking_days'] ?? '',
        'minimum_notice_hours' => $_POST['minimum_notice_hours'] ?? '',
        'max_party_size' => $_POST['max_party_size'] ?? '',
        'table_hold_time' => $_POST['table_hold_time'] ?? ''
    ];
    
    try {
        $db->beginTransaction();
        
        foreach ($settings as $key => $value) {
            $query = "INSERT INTO settings (setting_key, setting_value) VALUES (:key, :value) 
                     ON DUPLICATE KEY UPDATE setting_value = :value";
            $stmt = $db->prepare($query);
            $stmt->bindParam(':key', $key);
            $stmt->bindParam(':value', $value);
            $stmt->execute();
        }
        
        $db->commit();
        $success = 'Settings updated successfully!';
    } catch (Exception $e) {
        $db->rollBack();
        $error = 'Failed to update settings: ' . $e->getMessage();
    }
}

// Get current settings
$settings_query = "SELECT setting_key, setting_value FROM settings";
$settings_stmt = $db->prepare($settings_query);
$settings_stmt->execute();
$settings_data = $settings_stmt->fetchAll(PDO::FETCH_KEY_PAIR);

$page_title = 'Settings';
include '../includes/header.php';
?>

<div class="flex h-screen bg-gray-50">
    <?php include '../includes/sidebar.php'; ?>
    
    <div class="flex-1 flex flex-col overflow-hidden lg:ml-0">
        <?php include '../includes/dashboard_header.php'; ?>
        
        <main class="flex-1 overflow-y-auto p-6">
            <div class="space-y-6 fade-in">
                <div class="flex items-center justify-between">
                    <h1 class="text-2xl font-bold text-gray-900">Settings</h1>
                    <button type="submit" form="settingsForm" class="bg-amber-600 text-white px-4 py-2 rounded-lg hover:bg-amber-700 flex items-center hover-scale">
                        <i data-lucide="save" class="w-4 h-4 mr-2"></i>
                        Save Changes
                    </button>
                </div>

                <?php if ($success): ?>
                    <div class="bg-green-50 border border-green-200 text-green-600 px-4 py-3 rounded-lg alert-auto-hide">
                        <?php echo htmlspecialchars($success); ?>
                    </div>
                <?php endif; ?>

                <?php if ($error): ?>
                    <div class="bg-red-50 border border-red-200 text-red-600 px-4 py-3 rounded-lg alert-auto-hide">
                        <?php echo htmlspecialchars($error); ?>
                    </div>
                <?php endif; ?>

                <div class="bg-white rounded-lg shadow-sm border border-gray-200">
                    <!-- Tab Navigation -->
                    <div class="border-b border-gray-200">
                        <nav class="flex space-x-8 px-6">
                            <button onclick="showTab('restaurant')" id="restaurant-tab" class="py-4 px-1 border-b-2 font-medium text-sm transition-colors border-amber-500 text-amber-600">
                                <i data-lucide="utensils" class="w-4 h-4 inline mr-2"></i>
                                Restaurant Info
                            </button>
                            <button onclick="showTab('bookings')" id="bookings-tab" class="py-4 px-1 border-b-2 font-medium text-sm transition-colors border-transparent text-gray-500 hover:text-gray-700 hover:border-gray-300">
                                <i data-lucide="clock" class="w-4 h-4 inline mr-2"></i>
                                Booking Settings
                            </button>
                            <button onclick="showTab('notifications')" id="notifications-tab" class="py-4 px-1 border-b-2 font-medium text-sm transition-colors border-transparent text-gray-500 hover:text-gray-700 hover:border-gray-300">
                                <i data-lucide="bell" class="w-4 h-4 inline mr-2"></i>
                                Notifications
                            </button>
                        </nav>
                    </div>

                    <!-- Tab Content -->
                    <div class="p-6">
                        <form id="settingsForm" method="POST">
                            <!-- Restaurant Info Tab -->
                            <div id="restaurant-content" class="tab-content">
                                <div class="space-y-6">
                                    <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
                                        <div>
                                            <label class="block text-sm font-medium text-gray-700 mb-2">Restaurant Name</label>
                                            <input type="text" name="restaurant_name" 
                                                   value="<?php echo htmlspecialchars($settings_data['restaurant_name'] ?? 'Golden Fork'); ?>"
                                                   class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-amber-500 focus:border-amber-500">
                                        </div>
                                        <div>
                                            <label class="block text-sm font-medium text-gray-700 mb-2">Location</label>
                                            <input type="text" name="restaurant_location" 
                                                   value="<?php echo htmlspecialchars($settings_data['restaurant_location'] ?? 'Nakuru Maili Sita'); ?>"
                                                   class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-amber-500 focus:border-amber-500">
                                        </div>
                                    </div>

                                    <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
                                        <div>
                                            <label class="block text-sm font-medium text-gray-700 mb-2">Phone</label>
                                            <input type="tel" name="restaurant_phone" 
                                                   value="<?php echo htmlspecialchars($settings_data['restaurant_phone'] ?? '+254701234567'); ?>"
                                                   class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-amber-500 focus:border-amber-500">
                                        </div>
                                        <div>
                                            <label class="block text-sm font-medium text-gray-700 mb-2">Email</label>
                                            <input type="email" name="restaurant_email" 
                                                   value="<?php echo htmlspecialchars($settings_data['restaurant_email'] ?? 'info@goldenfork.com'); ?>"
                                                   class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-amber-500 focus:border-amber-500">
                                        </div>
                                    </div>

                                    <div>
                                        <h3 class="text-lg font-medium text-gray-900 mb-4">Operating Hours</h3>
                                        <div class="space-y-3">
                                            <?php
                                            $days = ['Monday', 'Tuesday', 'Wednesday', 'Thursday', 'Friday', 'Saturday', 'Sunday'];
                                            foreach ($days as $day):
                                            ?>
                                                <div class="flex items-center space-x-4">
                                                    <div class="w-20 text-sm font-medium text-gray-700"><?php echo $day; ?></div>
                                                    <input type="time" value="10:00" class="px-3 py-2 border border-gray-300 rounded-lg focus:ring-amber-500 focus:border-amber-500">
                                                    <span class="text-gray-500">to</span>
                                                    <input type="time" value="22:00" class="px-3 py-2 border border-gray-300 rounded-lg focus:ring-amber-500 focus:border-amber-500">
                                                    <label class="flex items-center">
                                                        <input type="checkbox" checked class="mr-2 text-amber-600 focus:ring-amber-500">
                                                        <span class="text-sm text-gray-600">Open</span>
                                                    </label>
                                                </div>
                                            <?php endforeach; ?>
                                        </div>
                                    </div>
                                </div>
                            </div>

                            <!-- Booking Settings Tab -->
                            <div id="bookings-content" class="tab-content hidden">
                                <div class="space-y-6">
                                    <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
                                        <div>
                                            <label class="block text-sm font-medium text-gray-700 mb-2">Advance Booking Days</label>
                                            <input type="number" name="advance_booking_days" 
                                                   value="<?php echo htmlspecialchars($settings_data['advance_booking_days'] ?? '30'); ?>"
                                                   class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-amber-500 focus:border-amber-500">
                                            <p class="text-sm text-gray-500 mt-1">How many days in advance customers can book</p>
                                        </div>
                                        <div>
                                            <label class="block text-sm font-medium text-gray-700 mb-2">Minimum Notice (hours)</label>
                                            <input type="number" name="minimum_notice_hours" 
                                                   value="<?php echo htmlspecialchars($settings_data['minimum_notice_hours'] ?? '2'); ?>"
                                                   class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-amber-500 focus:border-amber-500">
                                            <p class="text-sm text-gray-500 mt-1">Minimum time before reservation</p>
                                        </div>
                                    </div>

                                    <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
                                        <div>
                                            <label class="block text-sm font-medium text-gray-700 mb-2">Max Party Size</label>
                                            <input type="number" name="max_party_size" 
                                                   value="<?php echo htmlspecialchars($settings_data['max_party_size'] ?? '12'); ?>"
                                                   class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-amber-500 focus:border-amber-500">
                                        </div>
                                        <div>
                                            <label class="block text-sm font-medium text-gray-700 mb-2">Table Hold Time (minutes)</label>
                                            <input type="number" name="table_hold_time" 
                                                   value="<?php echo htmlspecialchars($settings_data['table_hold_time'] ?? '15'); ?>"
                                                   class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-amber-500 focus:border-amber-500">
                                        </div>
                                    </div>

                                    <div>
                                        <h3 class="text-lg font-medium text-gray-900 mb-4">Booking Policies</h3>
                                        <div class="space-y-3">
                                            <?php
                                            $policies = [
                                                ['label' => 'Require Phone Verification', 'description' => 'Verify customer phone numbers for reservations'],
                                                ['label' => 'Auto-confirm Bookings', 'description' => 'Automatically confirm reservations without admin approval'],
                                                ['label' => 'Send Reminder SMS', 'description' => 'Send SMS reminders 24 hours before reservation'],
                                                ['label' => 'Allow Online Cancellation', 'description' => 'Let customers cancel reservations online']
                                            ];
                                            foreach ($policies as $policy):
                                            ?>
                                                <div class="flex items-center justify-between p-4 border border-gray-200 rounded-lg">
                                                    <div>
                                                        <div class="font-medium text-gray-900"><?php echo $policy['label']; ?></div>
                                                        <div class="text-sm text-gray-500"><?php echo $policy['description']; ?></div>
                                                    </div>
                                                    <label class="flex items-center">
                                                        <input type="checkbox" checked class="mr-2 text-amber-600 focus:ring-amber-500">
                                                        <span class="text-sm">Enable</span>
                                                    </label>
                                                </div>
                                            <?php endforeach; ?>
                                        </div>
                                    </div>
                                </div>
                            </div>

                            <!-- Notifications Tab -->
                            <div id="notifications-content" class="tab-content hidden">
                                <div class="space-y-6">
                                    <div>
                                        <h3 class="text-lg font-medium text-gray-900 mb-4">Email Notifications</h3>
                                        <div class="space-y-3">
                                            <?php
                                            $email_notifications = [
                                                ['label' => 'New Reservations', 'description' => 'Get notified when customers make new reservations'],
                                                ['label' => 'Cancellations', 'description' => 'Receive alerts when reservations are cancelled'],
                                                ['label' => 'Reviews', 'description' => 'Get notified about new customer reviews'],
                                                ['label' => 'Daily Summary', 'description' => 'Receive daily reports at end of day']
                                            ];
                                            foreach ($email_notifications as $notification):
                                            ?>
                                                <div class="flex items-center justify-between p-4 border border-gray-200 rounded-lg">
                                                    <div>
                                                        <div class="font-medium text-gray-900"><?php echo $notification['label']; ?></div>
                                                        <div class="text-sm text-gray-500"><?php echo $notification['description']; ?></div>
                                                    </div>
                                                    <label class="flex items-center">
                                                        <input type="checkbox" checked class="mr-2 text-amber-600 focus:ring-amber-500">
                                                        <span class="text-sm">Enable</span>
                                                    </label>
                                                </div>
                                            <?php endforeach; ?>
                                        </div>
                                    </div>

                                    <div>
                                        <h3 class="text-lg font-medium text-gray-900 mb-4">SMS Notifications</h3>
                                        <div class="space-y-3">
                                            <?php
                                            $sms_notifications = [
                                                ['label' => 'Urgent Alerts', 'description' => 'Critical issues requiring immediate attention'],
                                                ['label' => 'Reservation Reminders', 'description' => 'Send SMS reminders to customers']
                                            ];
                                            foreach ($sms_notifications as $notification):
                                            ?>
                                                <div class="flex items-center justify-between p-4 border border-gray-200 rounded-lg">
                                                    <div>
                                                        <div class="font-medium text-gray-900"><?php echo $notification['label']; ?></div>
                                                        <div class="text-sm text-gray-500"><?php echo $notification['description']; ?></div>
                                                    </div>
                                                    <label class="flex items-center">
                                                        <input type="checkbox" checked class="mr-2 text-amber-600 focus:ring-amber-500">
                                                        <span class="text-sm">Enable</span>
                                                    </label>
                                                </div>
                                            <?php endforeach; ?>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </form>
                    </div>
                </div>
            </div>
        </main>
    </div>
</div>

<script>
function showTab(tabName) {
    // Hide all tab contents
    const contents = document.querySelectorAll('.tab-content');
    contents.forEach(content => content.classList.add('hidden'));
    
    // Remove active class from all tabs
    const tabs = document.querySelectorAll('[id$="-tab"]');
    tabs.forEach(tab => {
        tab.classList.remove('border-amber-500', 'text-amber-600');
        tab.classList.add('border-transparent', 'text-gray-500');
    });
    
    // Show selected tab content
    document.getElementById(tabName + '-content').classList.remove('hidden');
    
    // Add active class to selected tab
    const activeTab = document.getElementById(tabName + '-tab');
    activeTab.classList.remove('border-transparent', 'text-gray-500');
    activeTab.classList.add('border-amber-500', 'text-amber-600');
}
</script>

<?php include '../includes/footer.php'; ?>