<?php
require_once '../includes/auth.php';
require_once '../config/database.php';

requireAdmin();

$database = new Database();
$db = $database->getConnection();

$success = '';
$error = '';

// Handle table status updates
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action'])) {
    if ($_POST['action'] === 'update_status') {
        $table_id = $_POST['table_id'];
        $new_status = $_POST['status'];
        
        $update_query = "UPDATE restaurant_tables SET status = :status WHERE id = :id";
        $update_stmt = $db->prepare($update_query);
        $update_stmt->bindParam(':status', $new_status);
        $update_stmt->bindParam(':id', $table_id);
        
        if ($update_stmt->execute()) {
            $success = 'Table status updated successfully!';
        } else {
            $error = 'Failed to update table status.';
        }
    }
}

// Get all tables
$tables_query = "SELECT * FROM restaurant_tables ORDER BY table_number ASC";
$tables_stmt = $db->prepare($tables_query);
$tables_stmt->execute();
$tables = $tables_stmt->fetchAll(PDO::FETCH_ASSOC);

// Get table statistics
$stats_query = "
    SELECT 
        COUNT(*) as total_tables,
        SUM(CASE WHEN status = 'available' THEN 1 ELSE 0 END) as available_tables,
        SUM(CASE WHEN status = 'occupied' THEN 1 ELSE 0 END) as occupied_tables,
        SUM(CASE WHEN status = 'reserved' THEN 1 ELSE 0 END) as reserved_tables,
        SUM(CASE WHEN status = 'maintenance' THEN 1 ELSE 0 END) as maintenance_tables
    FROM restaurant_tables
";
$stats_stmt = $db->prepare($stats_query);
$stats_stmt->execute();
$stats = $stats_stmt->fetch(PDO::FETCH_ASSOC);

$page_title = 'Table Management';
include '../includes/header.php';
?>

<div class="flex h-screen bg-gray-50">
    <?php include '../includes/sidebar.php'; ?>
    
    <div class="flex-1 flex flex-col overflow-hidden lg:ml-0">
        <?php include '../includes/dashboard_header.php'; ?>
        
        <main class="flex-1 overflow-y-auto p-6">
            <div class="space-y-6 fade-in">
                <div class="flex items-center justify-between">
                    <h1 class="text-2xl font-bold text-gray-900">Table Management</h1>
                    <button onclick="showAddTableModal()" class="bg-amber-600 text-white px-4 py-2 rounded-lg hover:bg-amber-700 flex items-center hover-scale">
                        <i data-lucide="plus" class="w-4 h-4 mr-2"></i>
                        Add Table
                    </button>
                </div>

                <?php if ($success): ?>
                    <div class="bg-green-50 border border-green-200 text-green-600 px-4 py-3 rounded-lg alert-auto-hide">
                        <?php echo htmlspecialchars($success); ?>
                    </div>
                <?php endif; ?>

                <?php if ($error): ?>
                    <div class="bg-red-50 border border-red-200 text-red-600 px-4 py-3 rounded-lg alert-auto-hide">
                        <?php echo htmlspecialchars($error); ?>
                    </div>
                <?php endif; ?>

                <!-- Table Statistics -->
                <div class="bg-white rounded-lg shadow-sm border border-gray-200 p-6">
                    <h3 class="text-lg font-semibold text-gray-900 mb-4">Table Summary</h3>
                    <div class="grid grid-cols-2 md:grid-cols-5 gap-4">
                        <div class="text-center">
                            <div class="text-2xl font-bold text-blue-600"><?php echo $stats['total_tables']; ?></div>
                            <div class="text-sm text-gray-500">Total Tables</div>
                        </div>
                        <div class="text-center">
                            <div class="text-2xl font-bold text-green-600"><?php echo $stats['available_tables']; ?></div>
                            <div class="text-sm text-gray-500">Available</div>
                        </div>
                        <div class="text-center">
                            <div class="text-2xl font-bold text-red-600"><?php echo $stats['occupied_tables']; ?></div>
                            <div class="text-sm text-gray-500">Occupied</div>
                        </div>
                        <div class="text-center">
                            <div class="text-2xl font-bold text-yellow-600"><?php echo $stats['reserved_tables']; ?></div>
                            <div class="text-sm text-gray-500">Reserved</div>
                        </div>
                        <div class="text-center">
                            <div class="text-2xl font-bold text-gray-600"><?php echo $stats['maintenance_tables']; ?></div>
                            <div class="text-sm text-gray-500">Maintenance</div>
                        </div>
                    </div>
                </div>

                <!-- Table Grid View -->
                <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 xl:grid-cols-4 gap-6">
                    <?php foreach ($tables as $table): ?>
                        <?php
                        $status_colors = [
                            'available' => 'border-green-200 bg-green-50',
                            'occupied' => 'border-red-200 bg-red-50',
                            'reserved' => 'border-yellow-200 bg-yellow-50',
                            'maintenance' => 'border-gray-200 bg-gray-50'
                        ];
                        $status_icons = [
                            'available' => '🟢',
                            'occupied' => '🔴',
                            'reserved' => '🟡',
                            'maintenance' => '⚪'
                        ];
                        $border_class = $status_colors[$table['status']] ?? 'border-gray-200 bg-white';
                        $status_icon = $status_icons[$table['status']] ?? '⚪';
                        ?>
                        <div class="bg-white rounded-lg shadow-sm border-2 p-6 transition-all hover:shadow-md hover-scale <?php echo $border_class; ?>">
                            <div class="flex items-center justify-between mb-4">
                                <div class="flex items-center">
                                    <span class="text-2xl mr-2"><?php echo $status_icon; ?></span>
                                    <h3 class="text-lg font-semibold text-gray-900">Table <?php echo $table['table_number']; ?></h3>
                                </div>
                                <div class="flex space-x-1">
                                    <button onclick="editTable(<?php echo $table['id']; ?>)" class="p-1 text-gray-400 hover:text-amber-600">
                                        <i data-lucide="edit" class="w-4 h-4"></i>
                                    </button>
                                    <button onclick="deleteTable(<?php echo $table['id']; ?>)" class="p-1 text-gray-400 hover:text-red-600">
                                        <i data-lucide="trash-2" class="w-4 h-4"></i>
                                    </button>
                                </div>
                            </div>

                            <div class="space-y-3 mb-4">
                                <div class="flex items-center text-sm text-gray-600">
                                    <i data-lucide="users" class="w-4 h-4 mr-2"></i>
                                    Capacity: <?php echo $table['capacity']; ?> people
                                </div>
                                
                                <div class="flex items-center text-sm text-gray-600">
                                    <i data-lucide="map-pin" class="w-4 h-4 mr-2"></i>
                                    <?php echo htmlspecialchars($table['location']); ?>
                                </div>

                                <div class="pt-2">
                                    <?php
                                    $status_badge_colors = [
                                        'available' => 'bg-green-100 text-green-800',
                                        'occupied' => 'bg-red-100 text-red-800',
                                        'reserved' => 'bg-yellow-100 text-yellow-800',
                                        'maintenance' => 'bg-gray-100 text-gray-800'
                                    ];
                                    $badge_color = $status_badge_colors[$table['status']] ?? 'bg-gray-100 text-gray-800';
                                    ?>
                                    <span class="inline-flex px-2 py-1 text-xs font-semibold rounded-full <?php echo $badge_color; ?>">
                                        <?php echo ucfirst($table['status']); ?>
                                    </span>
                                </div>
                            </div>

                            <!-- Status Actions -->
                            <div class="pt-3 border-t border-gray-200">
                                <p class="text-xs text-gray-500 mb-2">Change Status:</p>
                                <div class="grid grid-cols-2 gap-1">
                                    <form method="POST" class="inline">
                                        <input type="hidden" name="action" value="update_status">
                                        <input type="hidden" name="table_id" value="<?php echo $table['id']; ?>">
                                        <input type="hidden" name="status" value="available">
                                        <button type="submit" class="w-full text-xs px-2 py-1 bg-green-50 text-green-700 rounded hover:bg-green-100">
                                            Available
                                        </button>
                                    </form>
                                    <form method="POST" class="inline">
                                        <input type="hidden" name="action" value="update_status">
                                        <input type="hidden" name="table_id" value="<?php echo $table['id']; ?>">
                                        <input type="hidden" name="status" value="occupied">
                                        <button type="submit" class="w-full text-xs px-2 py-1 bg-red-50 text-red-700 rounded hover:bg-red-100">
                                            Occupied
                                        </button>
                                    </form>
                                    <form method="POST" class="inline">
                                        <input type="hidden" name="action" value="update_status">
                                        <input type="hidden" name="table_id" value="<?php echo $table['id']; ?>">
                                        <input type="hidden" name="status" value="reserved">
                                        <button type="submit" class="w-full text-xs px-2 py-1 bg-yellow-50 text-yellow-700 rounded hover:bg-yellow-100">
                                            Reserved
                                        </button>
                                    </form>
                                    <form method="POST" class="inline">
                                        <input type="hidden" name="action" value="update_status">
                                        <input type="hidden" name="table_id" value="<?php echo $table['id']; ?>">
                                        <input type="hidden" name="status" value="maintenance">
                                        <button type="submit" class="w-full text-xs px-2 py-1 bg-gray-50 text-gray-700 rounded hover:bg-gray-100">
                                            Maintenance
                                        </button>
                                    </form>
                                </div>
                            </div>
                        </div>
                    <?php endforeach; ?>
                </div>
            </div>
        </main>
    </div>
</div>

<script>
function showAddTableModal() {
    alert('Add table modal would open here');
}

function editTable(id) {
    alert('Edit table modal would open for ID: ' + id);
}

function deleteTable(id) {
    if (confirm('Are you sure you want to delete this table?')) {
        alert('Delete table with ID: ' + id);
    }
}
</script>

<?php include '../includes/footer.php'; ?>