<?php
require_once '../includes/auth.php';
require_once '../config/database.php';

requireCustomer();

$database = new Database();
$db = $database->getConnection();
$user_id = $_SESSION['user_id'];

// Get customer statistics
$stats_query = "
    SELECT 
        (SELECT COUNT(*) FROM reservations WHERE customer_id = :user_id) as total_reservations,
        (SELECT COUNT(*) FROM reservations WHERE customer_id = :user_id AND status = 'confirmed' AND reservation_date >= CURDATE()) as upcoming_reservations,
        (SELECT COUNT(*) FROM reviews WHERE customer_id = :user_id) as total_reviews,
        (SELECT AVG(rating) FROM reviews WHERE customer_id = :user_id) as avg_rating
";
$stats_stmt = $db->prepare($stats_query);
$stats_stmt->bindParam(':user_id', $user_id);
$stats_stmt->execute();
$stats = $stats_stmt->fetch(PDO::FETCH_ASSOC);

// Get recent reservations
$recent_reservations_query = "
    SELECT r.*, rt.table_number, rt.location 
    FROM reservations r 
    JOIN restaurant_tables rt ON r.table_id = rt.id 
    WHERE r.customer_id = :user_id 
    ORDER BY r.created_at DESC 
    LIMIT 3
";
$recent_stmt = $db->prepare($recent_reservations_query);
$recent_stmt->bindParam(':user_id', $user_id);
$recent_stmt->execute();
$recent_reservations = $recent_stmt->fetchAll(PDO::FETCH_ASSOC);

// Get next reservation
$next_reservation_query = "
    SELECT r.*, rt.table_number, rt.location 
    FROM reservations r 
    JOIN restaurant_tables rt ON r.table_id = rt.id 
    WHERE r.customer_id = :user_id AND r.status = 'confirmed' AND r.reservation_date >= CURDATE()
    ORDER BY r.reservation_date ASC, r.reservation_time ASC 
    LIMIT 1
";
$next_stmt = $db->prepare($next_reservation_query);
$next_stmt->bindParam(':user_id', $user_id);
$next_stmt->execute();
$next_reservation = $next_stmt->fetch(PDO::FETCH_ASSOC);

$page_title = 'Customer Dashboard';
include '../includes/header.php';
?>

<div class="flex h-screen bg-gray-50">
    <?php include '../includes/sidebar.php'; ?>
    
    <div class="flex-1 flex flex-col overflow-hidden lg:ml-0">
        <?php include '../includes/dashboard_header.php'; ?>
        
        <main class="flex-1 overflow-y-auto p-6">
            <div class="space-y-6 fade-in">
                <div class="flex items-center justify-between">
                    <h1 class="text-2xl font-bold text-gray-900">Welcome back!</h1>
                    <div class="text-sm text-gray-500">
                        Member since: <?php echo date('F Y', strtotime(getCurrentUser()['created_at'])); ?>
                    </div>
                </div>

                <!-- Stats Grid -->
                <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6">
                    <div class="bg-white rounded-lg shadow-sm border border-gray-200 p-6 hover-scale">
                        <div class="flex items-center justify-between">
                            <div>
                                <p class="text-sm font-medium text-gray-600">Total Reservations</p>
                                <p class="text-2xl font-bold text-gray-900"><?php echo $stats['total_reservations'] ?: '0'; ?></p>
                                <p class="text-sm text-gray-500">All time</p>
                            </div>
                            <i data-lucide="calendar" class="w-8 h-8 text-blue-600"></i>
                        </div>
                    </div>

                    <div class="bg-white rounded-lg shadow-sm border border-gray-200 p-6 hover-scale">
                        <div class="flex items-center justify-between">
                            <div>
                                <p class="text-sm font-medium text-gray-600">Upcoming Bookings</p>
                                <p class="text-2xl font-bold text-gray-900"><?php echo $stats['upcoming_reservations'] ?: '0'; ?></p>
                                <p class="text-sm text-green-600">Confirmed</p>
                            </div>
                            <i data-lucide="clock" class="w-8 h-8 text-green-600"></i>
                        </div>
                    </div>

                    <div class="bg-white rounded-lg shadow-sm border border-gray-200 p-6 hover-scale">
                        <div class="flex items-center justify-between">
                            <div>
                                <p class="text-sm font-medium text-gray-600">Reviews Written</p>
                                <p class="text-2xl font-bold text-gray-900"><?php echo $stats['total_reviews'] ?: '0'; ?></p>
                                <p class="text-sm text-yellow-600">Thank you!</p>
                            </div>
                            <i data-lucide="star" class="w-8 h-8 text-yellow-600"></i>
                        </div>
                    </div>

                    <div class="bg-white rounded-lg shadow-sm border border-gray-200 p-6 hover-scale">
                        <div class="flex items-center justify-between">
                            <div>
                                <p class="text-sm font-medium text-gray-600">Average Rating</p>
                                <p class="text-2xl font-bold text-gray-900"><?php echo $stats['avg_rating'] ? number_format($stats['avg_rating'], 1) : 'N/A'; ?></p>
                                <p class="text-sm text-purple-600">Your ratings</p>
                            </div>
                            <i data-lucide="trending-up" class="w-8 h-8 text-purple-600"></i>
                        </div>
                    </div>
                </div>

                <!-- Next Reservation -->
                <?php if ($next_reservation): ?>
                <div class="bg-gradient-to-r from-amber-500 to-orange-500 rounded-lg p-6 text-white">
                    <h3 class="text-lg font-semibold mb-4">Your Next Reservation</h3>
                    <div class="grid grid-cols-1 md:grid-cols-4 gap-4">
                        <div class="flex items-center">
                            <i data-lucide="calendar" class="w-5 h-5 mr-2"></i>
                            <div>
                                <div class="font-medium"><?php echo date('M j, Y', strtotime($next_reservation['reservation_date'])); ?></div>
                                <div class="text-sm opacity-90">Date</div>
                            </div>
                        </div>
                        <div class="flex items-center">
                            <i data-lucide="clock" class="w-5 h-5 mr-2"></i>
                            <div>
                                <div class="font-medium"><?php echo date('g:i A', strtotime($next_reservation['reservation_time'])); ?></div>
                                <div class="text-sm opacity-90">Time</div>
                            </div>
                        </div>
                        <div class="flex items-center">
                            <i data-lucide="users" class="w-5 h-5 mr-2"></i>
                            <div>
                                <div class="font-medium">Table <?php echo $next_reservation['table_number']; ?></div>
                                <div class="text-sm opacity-90"><?php echo $next_reservation['location']; ?></div>
                            </div>
                        </div>
                        <div class="flex items-center">
                            <i data-lucide="user" class="w-5 h-5 mr-2"></i>
                            <div>
                                <div class="font-medium"><?php echo $next_reservation['guests']; ?> guests</div>
                                <div class="text-sm opacity-90">Party size</div>
                            </div>
                        </div>
                    </div>
                </div>
                <?php endif; ?>

                <!-- Recent Reservations & Quick Actions -->
                <div class="grid grid-cols-1 lg:grid-cols-2 gap-6">
                    <!-- Recent Reservations -->
                    <div class="bg-white rounded-lg shadow-sm border border-gray-200 p-6">
                        <h3 class="text-lg font-semibold text-gray-900 mb-4">Recent Reservations</h3>
                        <?php if (empty($recent_reservations)): ?>
                            <div class="text-center py-8">
                                <i data-lucide="calendar" class="w-12 h-12 text-gray-400 mx-auto mb-4"></i>
                                <p class="text-gray-500">No reservations yet</p>
                                <a href="/customer/reservations.php" class="text-amber-600 hover:text-amber-700 font-medium">Make your first reservation</a>
                            </div>
                        <?php else: ?>
                            <div class="space-y-4">
                                <?php foreach ($recent_reservations as $reservation): ?>
                                    <div class="flex items-center justify-between p-3 hover:bg-gray-50 rounded-lg">
                                        <div class="flex items-center space-x-3">
                                            <i data-lucide="calendar" class="w-5 h-5 text-gray-400"></i>
                                            <div>
                                                <p class="font-medium text-gray-900">
                                                    <?php echo date('M j, Y', strtotime($reservation['reservation_date'])); ?>
                                                </p>
                                                <p class="text-sm text-gray-500">
                                                    <?php echo date('g:i A', strtotime($reservation['reservation_time'])); ?> • 
                                                    Table <?php echo $reservation['table_number']; ?> • 
                                                    <?php echo $reservation['guests']; ?> guests
                                                </p>
                                            </div>
                                        </div>
                                        <?php
                                        $status_colors = [
                                            'pending' => 'bg-yellow-100 text-yellow-800',
                                            'confirmed' => 'bg-green-100 text-green-800',
                                            'completed' => 'bg-blue-100 text-blue-800',
                                            'cancelled' => 'bg-red-100 text-red-800'
                                        ];
                                        $color_class = $status_colors[$reservation['status']] ?? 'bg-gray-100 text-gray-800';
                                        ?>
                                        <span class="text-xs px-2 py-1 rounded-full <?php echo $color_class; ?>">
                                            <?php echo ucfirst($reservation['status']); ?>
                                        </span>
                                    </div>
                                <?php endforeach; ?>
                            </div>
                        <?php endif; ?>
                    </div>

                    <!-- Quick Actions -->
                    <div class="bg-white rounded-lg shadow-sm border border-gray-200 p-6">
                        <h3 class="text-lg font-semibold text-gray-900 mb-4">Quick Actions</h3>
                        <div class="grid grid-cols-2 gap-4">
                            <a href="/customer/reservations.php" class="bg-amber-50 hover:bg-amber-100 rounded-lg p-4 text-center transition-colors hover-scale">
                                <i data-lucide="calendar" class="w-6 h-6 mx-auto mb-2 text-amber-600"></i>
                                <span class="text-sm font-medium text-amber-700">Book Table</span>
                            </a>
                            <a href="/customer/reviews.php" class="bg-yellow-50 hover:bg-yellow-100 rounded-lg p-4 text-center transition-colors hover-scale">
                                <i data-lucide="star" class="w-6 h-6 mx-auto mb-2 text-yellow-600"></i>
                                <span class="text-sm font-medium text-yellow-700">Write Review</span>
                            </a>
                            <a href="/customer/tables.php" class="bg-blue-50 hover:bg-blue-100 rounded-lg p-4 text-center transition-colors hover-scale">
                                <i data-lucide="table" class="w-6 h-6 mx-auto mb-2 text-blue-600"></i>
                                <span class="text-sm font-medium text-blue-700">View Tables</span>
                            </a>
                            <a href="/customer/notifications.php" class="bg-purple-50 hover:bg-purple-100 rounded-lg p-4 text-center transition-colors hover-scale">
                                <i data-lucide="bell" class="w-6 h-6 mx-auto mb-2 text-purple-600"></i>
                                <span class="text-sm font-medium text-purple-700">Notifications</span>
                            </a>
                        </div>
                    </div>
                </div>

                <!-- Restaurant Info -->
                <div class="bg-white rounded-lg shadow-sm border border-gray-200 p-6">
                    <h3 class="text-lg font-semibold text-gray-900 mb-4">About Golden Fork</h3>
                    <div class="grid grid-cols-1 md:grid-cols-3 gap-6">
                        <div class="text-center">
                            <i data-lucide="map-pin" class="w-8 h-8 text-amber-600 mx-auto mb-2"></i>
                            <h4 class="font-medium text-gray-900">Location</h4>
                            <p class="text-sm text-gray-600">Nakuru Maili Sita</p>
                        </div>
                        <div class="text-center">
                            <i data-lucide="phone" class="w-8 h-8 text-amber-600 mx-auto mb-2"></i>
                            <h4 class="font-medium text-gray-900">Phone</h4>
                            <p class="text-sm text-gray-600">+254701234567</p>
                        </div>
                        <div class="text-center">
                            <i data-lucide="clock" class="w-8 h-8 text-amber-600 mx-auto mb-2"></i>
                            <h4 class="font-medium text-gray-900">Hours</h4>
                            <p class="text-sm text-gray-600">10:00 AM - 10:00 PM</p>
                        </div>
                    </div>
                </div>
            </div>
        </main>
    </div>
</div>

<?php include '../includes/footer.php'; ?>