<?php
require_once '../includes/auth.php';
require_once '../config/database.php';

requireCustomer();

$database = new Database();
$db = $database->getConnection();
$user_id = $_SESSION['user_id'];

$success = '';
$error = '';

// Handle mark as read
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action'])) {
    if ($_POST['action'] === 'mark_read') {
        $notification_id = $_POST['notification_id'];
        
        $update_query = "UPDATE notifications SET is_read = 1 WHERE id = :id AND user_id = :user_id";
        $update_stmt = $db->prepare($update_query);
        $update_stmt->bindParam(':id', $notification_id);
        $update_stmt->bindParam(':user_id', $user_id);
        
        if ($update_stmt->execute()) {
            $success = 'Notification marked as read.';
        } else {
            $error = 'Failed to update notification.';
        }
    } elseif ($_POST['action'] === 'delete') {
        $notification_id = $_POST['notification_id'];
        
        $delete_query = "DELETE FROM notifications WHERE id = :id AND user_id = :user_id";
        $delete_stmt = $db->prepare($delete_query);
        $delete_stmt->bindParam(':id', $notification_id);
        $delete_stmt->bindParam(':user_id', $user_id);
        
        if ($delete_stmt->execute()) {
            $success = 'Notification deleted.';
        } else {
            $error = 'Failed to delete notification.';
        }
    }
}

// Get filter parameters
$filter_status = $_GET['status'] ?? 'all';
$filter_type = $_GET['type'] ?? 'all';

// Build query with filters
$where_conditions = ['user_id = :user_id'];
$params = [':user_id' => $user_id];

if ($filter_status === 'read') {
    $where_conditions[] = 'is_read = 1';
} elseif ($filter_status === 'unread') {
    $where_conditions[] = 'is_read = 0';
}

if ($filter_type !== 'all') {
    $where_conditions[] = 'type = :type';
    $params[':type'] = $filter_type;
}

$where_clause = 'WHERE ' . implode(' AND ', $where_conditions);

// Get notifications
$notifications_query = "
    SELECT * FROM notifications 
    $where_clause
    ORDER BY created_at DESC
";

$notifications_stmt = $db->prepare($notifications_query);
foreach ($params as $key => $value) {
    $notifications_stmt->bindValue($key, $value);
}
$notifications_stmt->execute();
$notifications = $notifications_stmt->fetchAll(PDO::FETCH_ASSOC);

// Get notification statistics
$stats_query = "
    SELECT 
        COUNT(*) as total_notifications,
        COUNT(CASE WHEN is_read = 0 THEN 1 END) as unread_notifications,
        COUNT(CASE WHEN type = 'booking' THEN 1 END) as booking_notifications,
        COUNT(CASE WHEN type = 'reminder' THEN 1 END) as reminder_notifications
    FROM notifications 
    WHERE user_id = :user_id
";
$stats_stmt = $db->prepare($stats_query);
$stats_stmt->bindParam(':user_id', $user_id);
$stats_stmt->execute();
$stats = $stats_stmt->fetch(PDO::FETCH_ASSOC);

$page_title = 'Notifications';
include '../includes/header.php';
?>

<div class="flex h-screen bg-gray-50">
    <?php include '../includes/sidebar.php'; ?>
    
    <div class="flex-1 flex flex-col overflow-hidden lg:ml-0">
        <?php include '../includes/dashboard_header.php'; ?>
        
        <main class="flex-1 overflow-y-auto p-6">
            <div class="space-y-6 fade-in">
                <div class="flex items-center justify-between">
                    <h1 class="text-2xl font-bold text-gray-900">Notifications</h1>
                    <div class="flex items-center space-x-2">
                        <span class="text-sm text-gray-500">
                            <?php echo $stats['unread_notifications']; ?> unread
                        </span>
                        <i data-lucide="bell" class="w-5 h-5 text-gray-400"></i>
                    </div>
                </div>

                <?php if ($success): ?>
                    <div class="bg-green-50 border border-green-200 text-green-600 px-4 py-3 rounded-lg alert-auto-hide">
                        <?php echo htmlspecialchars($success); ?>
                    </div>
                <?php endif; ?>

                <?php if ($error): ?>
                    <div class="bg-red-50 border border-red-200 text-red-600 px-4 py-3 rounded-lg alert-auto-hide">
                        <?php echo htmlspecialchars($error); ?>
                    </div>
                <?php endif; ?>

                <!-- Notification Stats -->
                <div class="grid grid-cols-1 md:grid-cols-4 gap-6">
                    <div class="bg-white rounded-lg shadow-sm border border-gray-200 p-6 hover-scale">
                        <div class="flex items-center justify-between">
                            <div>
                                <div class="text-2xl font-bold text-blue-600"><?php echo $stats['total_notifications']; ?></div>
                                <div class="text-sm text-gray-500">Total</div>
                            </div>
                            <i data-lucide="bell" class="w-8 h-8 text-blue-600"></i>
                        </div>
                    </div>
                    <div class="bg-white rounded-lg shadow-sm border border-gray-200 p-6 hover-scale">
                        <div class="flex items-center justify-between">
                            <div>
                                <div class="text-2xl font-bold text-red-600"><?php echo $stats['unread_notifications']; ?></div>
                                <div class="text-sm text-gray-500">Unread</div>
                            </div>
                            <i data-lucide="message-square" class="w-8 h-8 text-red-600"></i>
                        </div>
                    </div>
                    <div class="bg-white rounded-lg shadow-sm border border-gray-200 p-6 hover-scale">
                        <div class="flex items-center justify-between">
                            <div>
                                <div class="text-2xl font-bold text-green-600"><?php echo $stats['booking_notifications']; ?></div>
                                <div class="text-sm text-gray-500">Booking Updates</div>
                            </div>
                            <i data-lucide="calendar" class="w-8 h-8 text-green-600"></i>
                        </div>
                    </div>
                    <div class="bg-white rounded-lg shadow-sm border border-gray-200 p-6 hover-scale">
                        <div class="flex items-center justify-between">
                            <div>
                                <div class="text-2xl font-bold text-purple-600"><?php echo $stats['reminder_notifications']; ?></div>
                                <div class="text-sm text-gray-500">Reminders</div>
                            </div>
                            <i data-lucide="bell" class="w-8 h-8 text-purple-600"></i>
                        </div>
                    </div>
                </div>

                <!-- Filters -->
                <div class="bg-white rounded-lg shadow-sm border border-gray-200 p-4">
                    <form method="GET" class="flex flex-col sm:flex-row gap-4">
                        <div class="flex items-center space-x-2">
                            <i data-lucide="filter" class="w-4 h-4 text-gray-400"></i>
                            <span class="text-sm font-medium text-gray-700">Filter by:</span>
                        </div>
                        
                        <div class="flex flex-wrap gap-2">
                            <select name="status" class="px-3 py-1 border border-gray-300 rounded-lg text-sm focus:ring-amber-500 focus:border-amber-500">
                                <option value="all" <?php echo $filter_status === 'all' ? 'selected' : ''; ?>>All Status</option>
                                <option value="unread" <?php echo $filter_status === 'unread' ? 'selected' : ''; ?>>Unread</option>
                                <option value="read" <?php echo $filter_status === 'read' ? 'selected' : ''; ?>>Read</option>
                            </select>
                            
                            <select name="type" class="px-3 py-1 border border-gray-300 rounded-lg text-sm focus:ring-amber-500 focus:border-amber-500">
                                <option value="all" <?php echo $filter_type === 'all' ? 'selected' : ''; ?>>All Types</option>
                                <option value="booking" <?php echo $filter_type === 'booking' ? 'selected' : ''; ?>>Booking Updates</option>
                                <option value="reminder" <?php echo $filter_type === 'reminder' ? 'selected' : ''; ?>>Reminders</option>
                                <option value="review" <?php echo $filter_type === 'review' ? 'selected' : ''; ?>>Reviews</option>
                                <option value="general" <?php echo $filter_type === 'general' ? 'selected' : ''; ?>>General</option>
                            </select>
                            
                            <button type="submit" class="bg-amber-600 text-white px-3 py-1 rounded-lg hover:bg-amber-700 text-sm">
                                Apply
                            </button>
                        </div>

                        <?php if ($filter_status !== 'all' || $filter_type !== 'all'): ?>
                            <a href="?" class="px-3 py-1 text-sm text-gray-600 hover:text-gray-800">
                                Clear filters
                            </a>
                        <?php endif; ?>
                    </form>
                </div>

                <!-- Notifications List -->
                <div class="space-y-3">
                    <?php if (empty($notifications)): ?>
                        <div class="text-center py-12">
                            <i data-lucide="bell" class="w-12 h-12 text-gray-400 mx-auto mb-4"></i>
                            <h3 class="text-lg font-medium text-gray-900 mb-2">No Notifications Found</h3>
                            <p class="text-gray-500">
                                <?php if ($filter_status === 'all' && $filter_type === 'all'): ?>
                                    You don't have any notifications yet.
                                <?php else: ?>
                                    No notifications match your current filters.
                                <?php endif; ?>
                            </p>
                        </div>
                    <?php else: ?>
                        <?php foreach ($notifications as $notification): ?>
                            <?php
                            $type_colors = [
                                'booking' => 'text-blue-600',
                                'review' => 'text-yellow-600',
                                'reminder' => 'text-purple-600',
                                'general' => 'text-gray-600'
                            ];
                            $type_icons = [
                                'booking' => 'calendar',
                                'review' => 'star',
                                'reminder' => 'bell',
                                'general' => 'message-square'
                            ];
                            $icon_color = $type_colors[$notification['type']] ?? 'text-gray-600';
                            $icon_name = $type_icons[$notification['type']] ?? 'message-square';
                            ?>
                            <div class="bg-white rounded-lg shadow-sm border p-4 transition-all hover:shadow-md hover-scale <?php echo $notification['is_read'] ? 'border-gray-200' : 'border-amber-200 bg-amber-50'; ?>">
                                <div class="flex items-start space-x-3">
                                    <div class="p-2 rounded-lg <?php echo $notification['is_read'] ? 'bg-gray-100' : 'bg-amber-100'; ?>">
                                        <i data-lucide="<?php echo $icon_name; ?>" class="w-5 h-5 <?php echo $icon_color; ?>"></i>
                                    </div>
                                    
                                    <div class="flex-1 min-w-0">
                                        <div class="flex items-center justify-between mb-1">
                                            <h3 class="font-medium <?php echo $notification['is_read'] ? 'text-gray-700' : 'text-gray-900'; ?>">
                                                <?php echo htmlspecialchars($notification['title']); ?>
                                            </h3>
                                            <div class="flex items-center space-x-2">
                                                <span class="text-sm text-gray-500">
                                                    <?php echo date('M j, Y g:i A', strtotime($notification['created_at'])); ?>
                                                </span>
                                                <?php if (!$notification['is_read']): ?>
                                                    <span class="w-2 h-2 bg-amber-500 rounded-full"></span>
                                                <?php endif; ?>
                                            </div>
                                        </div>
                                        
                                        <p class="text-sm mb-2 <?php echo $notification['is_read'] ? 'text-gray-500' : 'text-gray-700'; ?>">
                                            <?php echo htmlspecialchars($notification['message']); ?>
                                        </p>
                                        
                                        <div class="flex items-center justify-between">
                                            <?php
                                            $type_badge_colors = [
                                                'booking' => 'bg-blue-100 text-blue-800',
                                                'review' => 'bg-yellow-100 text-yellow-800',
                                                'reminder' => 'bg-purple-100 text-purple-800',
                                                'general' => 'bg-gray-100 text-gray-800'
                                            ];
                                            $badge_color = $type_badge_colors[$notification['type']] ?? 'bg-gray-100 text-gray-800';
                                            ?>
                                            <span class="inline-flex px-2 py-1 text-xs font-semibold rounded-full <?php echo $badge_color; ?>">
                                                <?php echo ucfirst($notification['type']); ?>
                                            </span>
                                            
                                            <div class="flex items-center space-x-2">
                                                <?php if (!$notification['is_read']): ?>
                                                    <form method="POST" class="inline">
                                                        <input type="hidden" name="action" value="mark_read">
                                                        <input type="hidden" name="notification_id" value="<?php echo $notification['id']; ?>">
                                                        <button type="submit" class="flex items-center px-2 py-1 text-xs text-green-600 hover:text-green-700 hover:bg-green-50 rounded">
                                                            <i data-lucide="check" class="w-3 h-3 mr-1"></i>
                                                            Mark as read
                                                        </button>
                                                    </form>
                                                <?php endif; ?>
                                                <form method="POST" class="inline">
                                                    <input type="hidden" name="action" value="delete">
                                                    <input type="hidden" name="notification_id" value="<?php echo $notification['id']; ?>">
                                                    <button type="submit" onclick="return confirm('Delete this notification?')" 
                                                            class="flex items-center px-2 py-1 text-xs text-red-600 hover:text-red-700 hover:bg-red-50 rounded">
                                                        <i data-lucide="x" class="w-3 h-3 mr-1"></i>
                                                        Delete
                                                    </button>
                                                </form>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        <?php endforeach; ?>
                    <?php endif; ?>
                </div>

                <!-- Notification Preferences -->
                <div class="bg-white rounded-lg shadow-sm border border-gray-200 p-6">
                    <h3 class="text-lg font-semibold text-gray-900 mb-4">Notification Preferences</h3>
                    <div class="space-y-3">
                        <?php
                        $preferences = [
                            ['label' => 'Booking confirmations', 'description' => 'Get notified when your reservations are confirmed'],
                            ['label' => 'Reservation reminders', 'description' => 'Receive reminders 24 hours before your booking'],
                            ['label' => 'Special offers', 'description' => 'Get notified about exclusive deals and promotions'],
                            ['label' => 'Review requests', 'description' => 'Receive requests to review your dining experience']
                        ];
                        foreach ($preferences as $pref):
                        ?>
                            <div class="flex items-center justify-between p-3 border border-gray-200 rounded-lg">
                                <div>
                                    <div class="font-medium text-gray-900"><?php echo $pref['label']; ?></div>
                                    <div class="text-sm text-gray-500"><?php echo $pref['description']; ?></div>
                                </div>
                                <label class="flex items-center">
                                    <input type="checkbox" checked class="mr-2 text-amber-600 focus:ring-amber-500">
                                    <span class="text-sm">Enable</span>
                                </label>
                            </div>
                        <?php endforeach; ?>
                    </div>
                </div>
            </div>
        </main>
    </div>
</div>

<?php include '../includes/footer.php'; ?>