<?php
require_once '../includes/auth.php';
require_once '../config/database.php';

requireCustomer();

$database = new Database();
$db = $database->getConnection();
$user_id = $_SESSION['user_id'];

$success = '';
$error = '';

// Handle new reservation
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action']) && $_POST['action'] === 'new_reservation') {
    $date = $_POST['date'];
    $time = $_POST['time'];
    $guests = $_POST['guests'];
    $special_requests = trim($_POST['special_requests']);
    
    if (empty($date) || empty($time) || empty($guests)) {
        $error = 'Please fill in all required fields.';
    } else {
        // Find available table
        $table_query = "
            SELECT id FROM restaurant_tables 
            WHERE capacity >= :guests AND status = 'available'
            AND id NOT IN (
                SELECT table_id FROM reservations 
                WHERE reservation_date = :date 
                AND reservation_time = :time 
                AND status IN ('confirmed', 'pending')
            )
            ORDER BY capacity ASC 
            LIMIT 1
        ";
        
        $table_stmt = $db->prepare($table_query);
        $table_stmt->bindParam(':guests', $guests);
        $table_stmt->bindParam(':date', $date);
        $table_stmt->bindParam(':time', $time);
        $table_stmt->execute();
        $available_table = $table_stmt->fetch(PDO::FETCH_ASSOC);
        
        if ($available_table) {
            // Create reservation
            $reservation_query = "
                INSERT INTO reservations (customer_id, table_id, reservation_date, reservation_time, guests, special_requests, status)
                VALUES (:customer_id, :table_id, :date, :time, :guests, :special_requests, 'pending')
            ";
            
            $reservation_stmt = $db->prepare($reservation_query);
            $reservation_stmt->bindParam(':customer_id', $user_id);
            $reservation_stmt->bindParam(':table_id', $available_table['id']);
            $reservation_stmt->bindParam(':date', $date);
            $reservation_stmt->bindParam(':time', $time);
            $reservation_stmt->bindParam(':guests', $guests);
            $reservation_stmt->bindParam(':special_requests', $special_requests);
            
            if ($reservation_stmt->execute()) {
                $success = 'Reservation submitted successfully! We will confirm it shortly.';
                
                // Create notification
                $notification_query = "INSERT INTO notifications (user_id, title, message, type) VALUES (:user_id, :title, :message, 'booking')";
                $notification_stmt = $db->prepare($notification_query);
                $notification_stmt->bindParam(':user_id', $user_id);
                $title = 'Reservation Submitted';
                $message = "Your reservation for $date at $time has been submitted and is pending confirmation.";
                $notification_stmt->bindParam(':title', $title);
                $notification_stmt->bindParam(':message', $message);
                $notification_stmt->execute();
            } else {
                $error = 'Failed to create reservation. Please try again.';
            }
        } else {
            $error = 'No tables available for the selected date, time, and party size.';
        }
    }
}

// Handle reservation cancellation
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action']) && $_POST['action'] === 'cancel_reservation') {
    $reservation_id = $_POST['reservation_id'];
    
    $cancel_query = "UPDATE reservations SET status = 'cancelled' WHERE id = :id AND customer_id = :customer_id";
    $cancel_stmt = $db->prepare($cancel_query);
    $cancel_stmt->bindParam(':id', $reservation_id);
    $cancel_stmt->bindParam(':customer_id', $user_id);
    
    if ($cancel_stmt->execute()) {
        $success = 'Reservation cancelled successfully.';
    } else {
        $error = 'Failed to cancel reservation.';
    }
}

// Get customer reservations
$reservations_query = "
    SELECT r.*, rt.table_number, rt.location 
    FROM reservations r 
    JOIN restaurant_tables rt ON r.table_id = rt.id 
    WHERE r.customer_id = :user_id 
    ORDER BY r.created_at DESC
";
$reservations_stmt = $db->prepare($reservations_query);
$reservations_stmt->bindParam(':user_id', $user_id);
$reservations_stmt->execute();
$reservations = $reservations_stmt->fetchAll(PDO::FETCH_ASSOC);

// Get reservation statistics
$stats_query = "
    SELECT 
        COUNT(*) as total_reservations,
        COUNT(CASE WHEN status = 'confirmed' THEN 1 END) as confirmed_reservations,
        COUNT(CASE WHEN status = 'pending' THEN 1 END) as pending_reservations,
        COUNT(CASE WHEN status = 'completed' THEN 1 END) as completed_reservations
    FROM reservations 
    WHERE customer_id = :user_id
";
$stats_stmt = $db->prepare($stats_query);
$stats_stmt->bindParam(':user_id', $user_id);
$stats_stmt->execute();
$stats = $stats_stmt->fetch(PDO::FETCH_ASSOC);

$page_title = 'My Reservations';
include '../includes/header.php';
?>

<div class="flex h-screen bg-gray-50">
    <?php include '../includes/sidebar.php'; ?>
    
    <div class="flex-1 flex flex-col overflow-hidden lg:ml-0">
        <?php include '../includes/dashboard_header.php'; ?>
        
        <main class="flex-1 overflow-y-auto p-6">
            <div class="space-y-6 fade-in">
                <div class="flex items-center justify-between">
                    <h1 class="text-2xl font-bold text-gray-900">My Reservations</h1>
                    <button onclick="showBookingModal()" class="bg-amber-600 text-white px-4 py-2 rounded-lg hover:bg-amber-700 flex items-center hover-scale">
                        <i data-lucide="plus" class="w-4 h-4 mr-2"></i>
                        New Reservation
                    </button>
                </div>

                <?php if ($success): ?>
                    <div class="bg-green-50 border border-green-200 text-green-600 px-4 py-3 rounded-lg alert-auto-hide">
                        <?php echo htmlspecialchars($success); ?>
                    </div>
                <?php endif; ?>

                <?php if ($error): ?>
                    <div class="bg-red-50 border border-red-200 text-red-600 px-4 py-3 rounded-lg alert-auto-hide">
                        <?php echo htmlspecialchars($error); ?>
                    </div>
                <?php endif; ?>

                <!-- Reservation Status Summary -->
                <div class="grid grid-cols-1 md:grid-cols-4 gap-6">
                    <div class="bg-white rounded-lg shadow-sm border border-gray-200 p-6 hover-scale">
                        <div class="text-2xl font-bold text-blue-600"><?php echo $stats['total_reservations']; ?></div>
                        <div class="text-sm text-gray-500">Total Reservations</div>
                    </div>
                    <div class="bg-white rounded-lg shadow-sm border border-gray-200 p-6 hover-scale">
                        <div class="text-2xl font-bold text-green-600"><?php echo $stats['confirmed_reservations']; ?></div>
                        <div class="text-sm text-gray-500">Confirmed</div>
                    </div>
                    <div class="bg-white rounded-lg shadow-sm border border-gray-200 p-6 hover-scale">
                        <div class="text-2xl font-bold text-yellow-600"><?php echo $stats['pending_reservations']; ?></div>
                        <div class="text-sm text-gray-500">Pending</div>
                    </div>
                    <div class="bg-white rounded-lg shadow-sm border border-gray-200 p-6 hover-scale">
                        <div class="text-2xl font-bold text-purple-600"><?php echo $stats['completed_reservations']; ?></div>
                        <div class="text-sm text-gray-500">Completed</div>
                    </div>
                </div>

                <!-- Reservations List -->
                <div class="space-y-4">
                    <?php if (empty($reservations)): ?>
                        <div class="text-center py-12">
                            <i data-lucide="calendar" class="w-12 h-12 text-gray-400 mx-auto mb-4"></i>
                            <h3 class="text-lg font-medium text-gray-900 mb-2">No Reservations Yet</h3>
                            <p class="text-gray-500 mb-4">Make your first reservation to get started!</p>
                            <button onclick="showBookingModal()" class="bg-amber-600 text-white px-6 py-2 rounded-lg hover:bg-amber-700">
                                Book Your First Table
                            </button>
                        </div>
                    <?php else: ?>
                        <?php foreach ($reservations as $reservation): ?>
                            <div class="bg-white rounded-lg shadow-sm border border-gray-200 p-6 hover-scale">
                                <div class="flex items-start justify-between">
                                    <div class="flex-1">
                                        <div class="flex items-center justify-between mb-4">
                                            <div class="flex items-center space-x-4">
                                                <div class="bg-amber-100 p-3 rounded-lg">
                                                    <i data-lucide="calendar" class="w-6 h-6 text-amber-600"></i>
                                                </div>
                                                <div>
                                                    <h3 class="text-lg font-semibold text-gray-900">
                                                        Table <?php echo $reservation['table_number']; ?>
                                                    </h3>
                                                    <p class="text-sm text-gray-500">Reservation #<?php echo $reservation['id']; ?></p>
                                                </div>
                                            </div>
                                            <?php
                                            $status_colors = [
                                                'pending' => 'bg-yellow-100 text-yellow-800',
                                                'confirmed' => 'bg-green-100 text-green-800',
                                                'completed' => 'bg-blue-100 text-blue-800',
                                                'cancelled' => 'bg-red-100 text-red-800'
                                            ];
                                            $color_class = $status_colors[$reservation['status']] ?? 'bg-gray-100 text-gray-800';
                                            ?>
                                            <span class="inline-flex px-3 py-1 text-sm font-semibold rounded-full <?php echo $color_class; ?>">
                                                <?php echo ucfirst($reservation['status']); ?>
                                            </span>
                                        </div>

                                        <div class="grid grid-cols-2 md:grid-cols-4 gap-4 mb-4">
                                            <div class="flex items-center text-sm text-gray-600">
                                                <i data-lucide="calendar" class="w-4 h-4 mr-2"></i>
                                                <?php echo date('M j, Y', strtotime($reservation['reservation_date'])); ?>
                                            </div>
                                            <div class="flex items-center text-sm text-gray-600">
                                                <i data-lucide="clock" class="w-4 h-4 mr-2"></i>
                                                <?php echo date('g:i A', strtotime($reservation['reservation_time'])); ?>
                                            </div>
                                            <div class="flex items-center text-sm text-gray-600">
                                                <i data-lucide="users" class="w-4 h-4 mr-2"></i>
                                                <?php echo $reservation['guests']; ?> guests
                                            </div>
                                            <div class="text-sm text-gray-500">
                                                <?php echo htmlspecialchars($reservation['location']); ?>
                                            </div>
                                        </div>

                                        <?php if ($reservation['special_requests']): ?>
                                            <div class="mb-4">
                                                <p class="text-sm font-medium text-gray-700 mb-1">Special Requests:</p>
                                                <p class="text-sm text-gray-600 bg-gray-50 p-2 rounded"><?php echo htmlspecialchars($reservation['special_requests']); ?></p>
                                            </div>
                                        <?php endif; ?>

                                        <div class="flex space-x-2">
                                            <?php if ($reservation['status'] === 'pending'): ?>
                                                <form method="POST" class="inline">
                                                    <input type="hidden" name="action" value="cancel_reservation">
                                                    <input type="hidden" name="reservation_id" value="<?php echo $reservation['id']; ?>">
                                                    <button type="submit" onclick="return confirm('Are you sure you want to cancel this reservation?')" 
                                                            class="flex items-center px-3 py-1 text-sm text-red-600 hover:text-red-700 hover:bg-red-50 rounded-lg transition-colors">
                                                        <i data-lucide="x" class="w-4 h-4 mr-1"></i>
                                                        Cancel
                                                    </button>
                                                </form>
                                            <?php elseif ($reservation['status'] === 'confirmed'): ?>
                                                <div class="flex items-center space-x-4 text-sm text-gray-600">
                                                    <div class="flex items-center">
                                                        <i data-lucide="phone" class="w-4 h-4 mr-1"></i>
                                                        Call if changes needed
                                                    </div>
                                                    <div class="flex items-center">
                                                        <i data-lucide="mail" class="w-4 h-4 mr-1"></i>
                                                        Confirmation sent
                                                    </div>
                                                </div>
                                            <?php endif; ?>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        <?php endforeach; ?>
                    <?php endif; ?>
                </div>
            </div>
        </main>
    </div>
</div>

<!-- Booking Modal -->
<div id="bookingModal" class="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50 hidden">
    <div class="bg-white rounded-lg p-6 w-full max-w-md mx-4">
        <h3 class="text-lg font-semibold text-gray-900 mb-4">Make a Reservation</h3>
        <form method="POST">
            <input type="hidden" name="action" value="new_reservation">
            
            <div class="space-y-4">
                <div>
                    <label class="block text-sm font-medium text-gray-700 mb-1">Date</label>
                    <input type="date" name="date" required
                           min="<?php echo date('Y-m-d'); ?>"
                           class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-amber-500 focus:border-amber-500">
                </div>
                
                <div>
                    <label class="block text-sm font-medium text-gray-700 mb-1">Time</label>
                    <select name="time" required class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-amber-500 focus:border-amber-500">
                        <option value="">Select time</option>
                        <?php
                        $times = ['17:00', '17:30', '18:00', '18:30', '19:00', '19:30', '20:00', '20:30', '21:00'];
                        foreach ($times as $time):
                        ?>
                            <option value="<?php echo $time; ?>"><?php echo date('g:i A', strtotime($time)); ?></option>
                        <?php endforeach; ?>
                    </select>
                </div>
                
                <div>
                    <label class="block text-sm font-medium text-gray-700 mb-1">Number of Guests</label>
                    <select name="guests" required class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-amber-500 focus:border-amber-500">
                        <?php for ($i = 1; $i <= 8; $i++): ?>
                            <option value="<?php echo $i; ?>"><?php echo $i; ?> <?php echo $i === 1 ? 'Guest' : 'Guests'; ?></option>
                        <?php endfor; ?>
                    </select>
                </div>
                
                <div>
                    <label class="block text-sm font-medium text-gray-700 mb-1">Special Requests</label>
                    <textarea name="special_requests" rows="3"
                              class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-amber-500 focus:border-amber-500"
                              placeholder="Any special requests or dietary requirements..."></textarea>
                </div>
            </div>
            
            <div class="flex space-x-3 mt-6">
                <button type="button" onclick="hideBookingModal()"
                        class="flex-1 px-4 py-2 border border-gray-300 text-gray-700 rounded-lg hover:bg-gray-50">
                    Cancel
                </button>
                <button type="submit"
                        class="flex-1 px-4 py-2 bg-amber-600 text-white rounded-lg hover:bg-amber-700">
                    Book Table
                </button>
            </div>
        </form>
    </div>
</div>

<script>
function showBookingModal() {
    document.getElementById('bookingModal').classList.remove('hidden');
}

function hideBookingModal() {
    document.getElementById('bookingModal').classList.add('hidden');
}
</script>

<?php include '../includes/footer.php'; ?>