<?php
require_once '../includes/auth.php';
require_once '../config/database.php';

requireCustomer();

$database = new Database();
$db = $database->getConnection();
$user_id = $_SESSION['user_id'];

$success = '';
$error = '';

// Handle new review submission
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action']) && $_POST['action'] === 'submit_review') {
    $rating = $_POST['rating'];
    $comment = trim($_POST['comment']);
    $reservation_id = $_POST['reservation_id'] ?: null;
    
    if (empty($rating) || empty($comment)) {
        $error = 'Please provide both a rating and comment.';
    } elseif ($rating < 1 || $rating > 5) {
        $error = 'Rating must be between 1 and 5 stars.';
    } else {
        $review_query = "
            INSERT INTO reviews (customer_id, reservation_id, rating, comment, status)
            VALUES (:customer_id, :reservation_id, :rating, :comment, 'pending')
        ";
        
        $review_stmt = $db->prepare($review_query);
        $review_stmt->bindParam(':customer_id', $user_id);
        $review_stmt->bindParam(':reservation_id', $reservation_id);
        $review_stmt->bindParam(':rating', $rating);
        $review_stmt->bindParam(':comment', $comment);
        
        if ($review_stmt->execute()) {
            $success = 'Review submitted successfully! It will be published after moderation.';
            
            // Create notification
            $notification_query = "INSERT INTO notifications (user_id, title, message, type) VALUES (:user_id, :title, :message, 'review')";
            $notification_stmt = $db->prepare($notification_query);
            $notification_stmt->bindParam(':user_id', $user_id);
            $title = 'Review Submitted';
            $message = 'Thank you for your review! It will be published after moderation.';
            $notification_stmt->bindParam(':title', $title);
            $notification_stmt->bindParam(':message', $message);
            $notification_stmt->execute();
        } else {
            $error = 'Failed to submit review. Please try again.';
        }
    }
}

// Get customer reviews
$reviews_query = "
    SELECT r.*, res.reservation_date, res.reservation_time, rt.table_number
    FROM reviews r
    LEFT JOIN reservations res ON r.reservation_id = res.id
    LEFT JOIN restaurant_tables rt ON res.table_id = rt.id
    WHERE r.customer_id = :user_id
    ORDER BY r.created_at DESC
";
$reviews_stmt = $db->prepare($reviews_query);
$reviews_stmt->bindParam(':user_id', $user_id);
$reviews_stmt->execute();
$reviews = $reviews_stmt->fetchAll(PDO::FETCH_ASSOC);

// Get completed reservations for review options
$reservations_query = "
    SELECT r.id, r.reservation_date, r.reservation_time, rt.table_number
    FROM reservations r
    JOIN restaurant_tables rt ON r.table_id = rt.id
    WHERE r.customer_id = :user_id AND r.status = 'completed'
    AND r.id NOT IN (SELECT reservation_id FROM reviews WHERE customer_id = :user_id AND reservation_id IS NOT NULL)
    ORDER BY r.reservation_date DESC
";
$reservations_stmt = $db->prepare($reservations_query);
$reservations_stmt->bindParam(':user_id', $user_id);
$reservations_stmt->execute();
$available_reservations = $reservations_stmt->fetchAll(PDO::FETCH_ASSOC);

// Get review statistics
$stats_query = "
    SELECT 
        COUNT(*) as total_reviews,
        AVG(rating) as avg_rating,
        COUNT(CASE WHEN status = 'approved' THEN 1 END) as approved_reviews
    FROM reviews 
    WHERE customer_id = :user_id
";
$stats_stmt = $db->prepare($stats_query);
$stats_stmt->bindParam(':user_id', $user_id);
$stats_stmt->execute();
$stats = $stats_stmt->fetch(PDO::FETCH_ASSOC);

$page_title = 'My Reviews';
include '../includes/header.php';
?>

<div class="flex h-screen bg-gray-50">
    <?php include '../includes/sidebar.php'; ?>
    
    <div class="flex-1 flex flex-col overflow-hidden lg:ml-0">
        <?php include '../includes/dashboard_header.php'; ?>
        
        <main class="flex-1 overflow-y-auto p-6">
            <div class="space-y-6 fade-in">
                <div class="flex items-center justify-between">
                    <h1 class="text-2xl font-bold text-gray-900">My Reviews</h1>
                    <button onclick="showReviewModal()" class="bg-amber-600 text-white px-4 py-2 rounded-lg hover:bg-amber-700 flex items-center hover-scale">
                        <i data-lucide="plus" class="w-4 h-4 mr-2"></i>
                        Write Review
                    </button>
                </div>

                <?php if ($success): ?>
                    <div class="bg-green-50 border border-green-200 text-green-600 px-4 py-3 rounded-lg alert-auto-hide">
                        <?php echo htmlspecialchars($success); ?>
                    </div>
                <?php endif; ?>

                <?php if ($error): ?>
                    <div class="bg-red-50 border border-red-200 text-red-600 px-4 py-3 rounded-lg alert-auto-hide">
                        <?php echo htmlspecialchars($error); ?>
                    </div>
                <?php endif; ?>

                <!-- Review Stats -->
                <div class="grid grid-cols-1 md:grid-cols-3 gap-6">
                    <div class="bg-white rounded-lg shadow-sm border border-gray-200 p-6 hover-scale">
                        <div class="text-2xl font-bold text-amber-600"><?php echo $stats['total_reviews']; ?></div>
                        <div class="text-sm text-gray-500">Reviews Written</div>
                    </div>
                    <div class="bg-white rounded-lg shadow-sm border border-gray-200 p-6 hover-scale">
                        <div class="text-2xl font-bold text-yellow-600">
                            <?php echo $stats['total_reviews'] > 0 ? number_format($stats['avg_rating'], 1) : 'N/A'; ?>
                        </div>
                        <div class="text-sm text-gray-500">Average Rating Given</div>
                    </div>
                    <div class="bg-white rounded-lg shadow-sm border border-gray-200 p-6 hover-scale">
                        <div class="text-2xl font-bold text-green-600"><?php echo $stats['approved_reviews']; ?></div>
                        <div class="text-sm text-gray-500">Approved Reviews</div>
                    </div>
                </div>

                <!-- Reviews List -->
                <div class="space-y-4">
                    <?php if (empty($reviews)): ?>
                        <div class="text-center py-12">
                            <i data-lucide="star" class="w-12 h-12 text-gray-400 mx-auto mb-4"></i>
                            <h3 class="text-lg font-medium text-gray-900 mb-2">No Reviews Yet</h3>
                            <p class="text-gray-500 mb-4">Share your dining experience with others!</p>
                            <button onclick="showReviewModal()" class="bg-amber-600 text-white px-6 py-2 rounded-lg hover:bg-amber-700">
                                Write Your First Review
                            </button>
                        </div>
                    <?php else: ?>
                        <?php foreach ($reviews as $review): ?>
                            <div class="bg-white rounded-lg shadow-sm border border-gray-200 p-6 hover-scale">
                                <div class="flex items-start justify-between mb-4">
                                    <div class="flex items-center space-x-3">
                                        <div class="bg-amber-100 p-2 rounded-lg">
                                            <i data-lucide="star" class="w-6 h-6 text-amber-600"></i>
                                        </div>
                                        <div>
                                            <div class="flex items-center space-x-2">
                                                <div class="flex">
                                                    <?php for ($i = 1; $i <= 5; $i++): ?>
                                                        <i data-lucide="star" class="w-4 h-4 <?php echo $i <= $review['rating'] ? 'text-yellow-400 fill-current' : 'text-gray-300'; ?>"></i>
                                                    <?php endfor; ?>
                                                </div>
                                                <span class="text-sm text-gray-500">(<?php echo $review['rating']; ?>/5)</span>
                                            </div>
                                            <p class="text-sm text-gray-500 flex items-center mt-1">
                                                <i data-lucide="calendar" class="w-4 h-4 mr-1"></i>
                                                <?php echo date('M j, Y', strtotime($review['created_at'])); ?>
                                            </p>
                                        </div>
                                    </div>
                                    <?php
                                    $status_colors = [
                                        'pending' => 'bg-yellow-100 text-yellow-800',
                                        'approved' => 'bg-green-100 text-green-800',
                                        'rejected' => 'bg-red-100 text-red-800'
                                    ];
                                    $color_class = $status_colors[$review['status']] ?? 'bg-gray-100 text-gray-800';
                                    ?>
                                    <span class="inline-flex px-2 py-1 text-xs font-semibold rounded-full <?php echo $color_class; ?>">
                                        <?php echo ucfirst($review['status']); ?>
                                    </span>
                                </div>

                                <p class="text-gray-700 mb-4"><?php echo htmlspecialchars($review['comment']); ?></p>

                                <?php if ($review['reservation_id']): ?>
                                    <p class="text-sm text-gray-500 bg-gray-50 p-2 rounded">
                                        Related to reservation on <?php echo date('M j, Y', strtotime($review['reservation_date'])); ?> 
                                        at Table <?php echo $review['table_number']; ?>
                                    </p>
                                <?php endif; ?>

                                <?php if ($review['status'] === 'pending'): ?>
                                    <div class="mt-4 p-3 bg-yellow-50 border border-yellow-200 rounded-lg">
                                        <p class="text-sm text-yellow-800">
                                            Your review is being moderated and will be published soon.
                                        </p>
                                    </div>
                                <?php elseif ($review['status'] === 'approved'): ?>
                                    <div class="mt-4 p-3 bg-green-50 border border-green-200 rounded-lg">
                                        <p class="text-sm text-green-800">
                                            Thank you! Your review has been published and helps other customers.
                                        </p>
                                    </div>
                                <?php endif; ?>
                            </div>
                        <?php endforeach; ?>
                    <?php endif; ?>
                </div>

                <!-- Review Guidelines -->
                <div class="bg-blue-50 border border-blue-200 rounded-lg p-4">
                    <h4 class="font-medium text-blue-900 mb-2">Review Guidelines</h4>
                    <ul class="text-sm text-blue-800 space-y-1">
                        <li>• Be honest and constructive in your feedback</li>
                        <li>• Focus on your dining experience, food quality, and service</li>
                        <li>• Avoid personal attacks or inappropriate language</li>
                        <li>• Reviews are moderated and may take 24-48 hours to appear</li>
                    </ul>
                </div>
            </div>
        </main>
    </div>
</div>

<!-- Review Modal -->
<div id="reviewModal" class="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50 hidden">
    <div class="bg-white rounded-lg p-6 w-full max-w-md mx-4">
        <h3 class="text-lg font-semibold text-gray-900 mb-4">Write a Review</h3>
        <form method="POST">
            <input type="hidden" name="action" value="submit_review">
            
            <div class="space-y-4">
                <div>
                    <label class="block text-sm font-medium text-gray-700 mb-2">Rating</label>
                    <div class="flex space-x-1" id="rating-stars">
                        <?php for ($i = 1; $i <= 5; $i++): ?>
                            <i data-lucide="star" class="w-6 h-6 text-gray-300 cursor-pointer hover:text-yellow-400" 
                               onclick="setRating(<?php echo $i; ?>)" data-rating="<?php echo $i; ?>"></i>
                        <?php endfor; ?>
                    </div>
                    <input type="hidden" name="rating" id="rating-input" required>
                    <p class="text-sm text-gray-500 mt-1">Click stars to rate your experience</p>
                </div>
                
                <div>
                    <label class="block text-sm font-medium text-gray-700 mb-1">Related Reservation (Optional)</label>
                    <select name="reservation_id" class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-amber-500 focus:border-amber-500">
                        <option value="">Select a reservation</option>
                        <?php foreach ($available_reservations as $reservation): ?>
                            <option value="<?php echo $reservation['id']; ?>">
                                <?php echo date('M j, Y', strtotime($reservation['reservation_date'])); ?> - 
                                Table <?php echo $reservation['table_number']; ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                </div>

                <div>
                    <label class="block text-sm font-medium text-gray-700 mb-1">Your Review</label>
                    <textarea name="comment" rows="4" required
                              class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-amber-500 focus:border-amber-500"
                              placeholder="Share your experience at Golden Fork..."></textarea>
                </div>
            </div>
            
            <div class="flex space-x-3 mt-6">
                <button type="button" onclick="hideReviewModal()"
                        class="flex-1 px-4 py-2 border border-gray-300 text-gray-700 rounded-lg hover:bg-gray-50">
                    Cancel
                </button>
                <button type="submit"
                        class="flex-1 px-4 py-2 bg-amber-600 text-white rounded-lg hover:bg-amber-700">
                    Submit Review
                </button>
            </div>
        </form>
    </div>
</div>

<script>
let currentRating = 0;

function showReviewModal() {
    document.getElementById('reviewModal').classList.remove('hidden');
}

function hideReviewModal() {
    document.getElementById('reviewModal').classList.add('hidden');
    resetRating();
}

function setRating(rating) {
    currentRating = rating;
    document.getElementById('rating-input').value = rating;
    
    const stars = document.querySelectorAll('#rating-stars i');
    stars.forEach((star, index) => {
        if (index < rating) {
            star.classList.remove('text-gray-300');
            star.classList.add('text-yellow-400', 'fill-current');
        } else {
            star.classList.remove('text-yellow-400', 'fill-current');
            star.classList.add('text-gray-300');
        }
    });
}

function resetRating() {
    currentRating = 0;
    document.getElementById('rating-input').value = '';
    
    const stars = document.querySelectorAll('#rating-stars i');
    stars.forEach(star => {
        star.classList.remove('text-yellow-400', 'fill-current');
        star.classList.add('text-gray-300');
    });
}
</script>

<?php include '../includes/footer.php'; ?>