<?php
require_once '../includes/auth.php';
require_once '../config/database.php';

requireCustomer();

$database = new Database();
$db = $database->getConnection();
$user_id = $_SESSION['user_id'];

$success = '';
$error = '';

// Handle profile update
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $name = trim($_POST['name']);
    $email = trim($_POST['email']);
    $phone = trim($_POST['phone']);
    $current_password = $_POST['current_password'] ?? '';
    $new_password = $_POST['new_password'] ?? '';
    $confirm_password = $_POST['confirm_password'] ?? '';
    
    if (empty($name) || empty($email)) {
        $error = 'Name and email are required.';
    } elseif (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
        $error = 'Please enter a valid email address.';
    } else {
        // Check if email is already taken by another user
        $email_check_query = "SELECT id FROM users WHERE email = :email AND id != :user_id";
        $email_check_stmt = $db->prepare($email_check_query);
        $email_check_stmt->bindParam(':email', $email);
        $email_check_stmt->bindParam(':user_id', $user_id);
        $email_check_stmt->execute();
        
        if ($email_check_stmt->rowCount() > 0) {
            $error = 'This email is already in use by another account.';
        } else {
            // Update basic profile info
            $update_query = "UPDATE users SET name = :name, email = :email, phone = :phone WHERE id = :user_id";
            $update_stmt = $db->prepare($update_query);
            $update_stmt->bindParam(':name', $name);
            $update_stmt->bindParam(':email', $email);
            $update_stmt->bindParam(':phone', $phone);
            $update_stmt->bindParam(':user_id', $user_id);
            
            if ($update_stmt->execute()) {
                $_SESSION['user_name'] = $name;
                $_SESSION['user_email'] = $email;
                
                // Handle password change if provided
                if (!empty($new_password)) {
                    if (empty($current_password)) {
                        $error = 'Current password is required to set a new password.';
                    } elseif (strlen($new_password) < 6) {
                        $error = 'New password must be at least 6 characters long.';
                    } elseif ($new_password !== $confirm_password) {
                        $error = 'New passwords do not match.';
                    } else {
                        // Verify current password
                        $password_check_query = "SELECT password FROM users WHERE id = :user_id";
                        $password_check_stmt = $db->prepare($password_check_query);
                        $password_check_stmt->bindParam(':user_id', $user_id);
                        $password_check_stmt->execute();
                        $user_data = $password_check_stmt->fetch(PDO::FETCH_ASSOC);
                        
                        if (password_verify($current_password, $user_data['password'])) {
                            $hashed_password = password_hash($new_password, PASSWORD_DEFAULT);
                            $password_update_query = "UPDATE users SET password = :password WHERE id = :user_id";
                            $password_update_stmt = $db->prepare($password_update_query);
                            $password_update_stmt->bindParam(':password', $hashed_password);
                            $password_update_stmt->bindParam(':user_id', $user_id);
                            
                            if ($password_update_stmt->execute()) {
                                $success = 'Profile and password updated successfully!';
                            } else {
                                $error = 'Profile updated but failed to change password.';
                            }
                        } else {
                            $error = 'Current password is incorrect.';
                        }
                    }
                } else {
                    $success = 'Profile updated successfully!';
                }
            } else {
                $error = 'Failed to update profile. Please try again.';
            }
        }
    }
}

// Get current user data
$user_query = "SELECT * FROM users WHERE id = :user_id";
$user_stmt = $db->prepare($user_query);
$user_stmt->bindParam(':user_id', $user_id);
$user_stmt->execute();
$user_data = $user_stmt->fetch(PDO::FETCH_ASSOC);

$page_title = 'Account Settings';
include '../includes/header.php';
?>

<div class="flex h-screen bg-gray-50">
    <?php include '../includes/sidebar.php'; ?>
    
    <div class="flex-1 flex flex-col overflow-hidden lg:ml-0">
        <?php include '../includes/dashboard_header.php'; ?>
        
        <main class="flex-1 overflow-y-auto p-6">
            <div class="space-y-6 fade-in">
                <div class="flex items-center justify-between">
                    <h1 class="text-2xl font-bold text-gray-900">Account Settings</h1>
                    <button type="submit" form="settingsForm" class="bg-amber-600 text-white px-4 py-2 rounded-lg hover:bg-amber-700 flex items-center hover-scale">
                        <i data-lucide="save" class="w-4 h-4 mr-2"></i>
                        Save Changes
                    </button>
                </div>

                <?php if ($success): ?>
                    <div class="bg-green-50 border border-green-200 text-green-600 px-4 py-3 rounded-lg alert-auto-hide">
                        <?php echo htmlspecialchars($success); ?>
                    </div>
                <?php endif; ?>

                <?php if ($error): ?>
                    <div class="bg-red-50 border border-red-200 text-red-600 px-4 py-3 rounded-lg alert-auto-hide">
                        <?php echo htmlspecialchars($error); ?>
                    </div>
                <?php endif; ?>

                <div class="bg-white rounded-lg shadow-sm border border-gray-200">
                    <!-- Tab Navigation -->
                    <div class="border-b border-gray-200">
                        <nav class="flex space-x-8 px-6">
                            <button onclick="showTab('profile')" id="profile-tab" class="py-4 px-1 border-b-2 font-medium text-sm transition-colors border-amber-500 text-amber-600">
                                <i data-lucide="user" class="w-4 h-4 inline mr-2"></i>
                                Profile
                            </button>
                            <button onclick="showTab('notifications')" id="notifications-tab" class="py-4 px-1 border-b-2 font-medium text-sm transition-colors border-transparent text-gray-500 hover:text-gray-700 hover:border-gray-300">
                                <i data-lucide="bell" class="w-4 h-4 inline mr-2"></i>
                                Notifications
                            </button>
                        </nav>
                    </div>

                    <!-- Tab Content -->
                    <div class="p-6">
                        <form id="settingsForm" method="POST">
                            <!-- Profile Tab -->
                            <div id="profile-content" class="tab-content">
                                <div class="space-y-6">
                                    <div>
                                        <h3 class="text-lg font-medium text-gray-900 mb-4">Personal Information</h3>
                                        <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
                                            <div>
                                                <label class="block text-sm font-medium text-gray-700 mb-2">Full Name</label>
                                                <input type="text" name="name" value="<?php echo htmlspecialchars($user_data['name']); ?>" required
                                                       class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-amber-500 focus:border-amber-500">
                                            </div>
                                            <div>
                                                <label class="block text-sm font-medium text-gray-700 mb-2">Email Address</label>
                                                <input type="email" name="email" value="<?php echo htmlspecialchars($user_data['email']); ?>" required
                                                       class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-amber-500 focus:border-amber-500">
                                            </div>
                                        </div>
                                    </div>

                                    <div>
                                        <h3 class="text-lg font-medium text-gray-900 mb-4">Contact Information</h3>
                                        <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
                                            <div>
                                                <label class="block text-sm font-medium text-gray-700 mb-2">Phone Number</label>
                                                <div class="relative">
                                                    <i data-lucide="phone" class="absolute left-3 top-3 w-4 h-4 text-gray-400"></i>
                                                    <input type="tel" name="phone" value="<?php echo htmlspecialchars($user_data['phone'] ?? ''); ?>"
                                                           class="w-full pl-10 pr-3 py-2 border border-gray-300 rounded-lg focus:ring-amber-500 focus:border-amber-500"
                                                           placeholder="+254701234567">
                                                </div>
                                            </div>
                                        </div>
                                    </div>

                                    <div>
                                        <h3 class="text-lg font-medium text-gray-900 mb-4">Change Password</h3>
                                        <div class="space-y-4">
                                            <div>
                                                <label class="block text-sm font-medium text-gray-700 mb-2">Current Password</label>
                                                <input type="password" name="current_password"
                                                       class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-amber-500 focus:border-amber-500"
                                                       placeholder="Enter current password">
                                            </div>
                                            
                                            <div class="grid grid-cols-1 md:grid-cols-2 gap-4">
                                                <div>
                                                    <label class="block text-sm font-medium text-gray-700 mb-2">New Password</label>
                                                    <input type="password" name="new_password"
                                                           class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-amber-500 focus:border-amber-500"
                                                           placeholder="Enter new password">
                                                </div>
                                                <div>
                                                    <label class="block text-sm font-medium text-gray-700 mb-2">Confirm New Password</label>
                                                    <input type="password" name="confirm_password"
                                                           class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-amber-500 focus:border-amber-500"
                                                           placeholder="Confirm new password">
                                                </div>
                                            </div>
                                            <p class="text-sm text-gray-500">Leave password fields empty if you don't want to change your password.</p>
                                        </div>
                                    </div>

                                    <div>
                                        <h3 class="text-lg font-medium text-gray-900 mb-4">Account Information</h3>
                                        <div class="bg-gray-50 rounded-lg p-4">
                                            <div class="grid grid-cols-1 md:grid-cols-2 gap-4 text-sm">
                                                <div>
                                                    <span class="font-medium text-gray-700">Account Type:</span>
                                                    <span class="text-gray-600 ml-2 capitalize"><?php echo $user_data['role']; ?></span>
                                                </div>
                                                <div>
                                                    <span class="font-medium text-gray-700">Member Since:</span>
                                                    <span class="text-gray-600 ml-2"><?php echo date('M j, Y', strtotime($user_data['created_at'])); ?></span>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>

                            <!-- Notifications Tab -->
                            <div id="notifications-content" class="tab-content hidden">
                                <div class="space-y-6">
                                    <div>
                                        <h3 class="text-lg font-medium text-gray-900 mb-4">Email Notifications</h3>
                                        <div class="space-y-3">
                                            <?php
                                            $email_notifications = [
                                                ['label' => 'Booking Confirmations', 'description' => 'Get email confirmation when your reservation is confirmed'],
                                                ['label' => 'Booking Reminders', 'description' => 'Receive reminder emails 24 hours before your reservation'],
                                                ['label' => 'Special Offers', 'description' => 'Be notified about exclusive deals and promotions'],
                                                ['label' => 'Menu Updates', 'description' => 'Get notified when new dishes are added to the menu'],
                                                ['label' => 'Review Requests', 'description' => 'Receive requests to review your dining experience']
                                            ];
                                            foreach ($email_notifications as $notification):
                                            ?>
                                                <div class="flex items-center justify-between p-4 border border-gray-200 rounded-lg">
                                                    <div>
                                                        <div class="font-medium text-gray-900"><?php echo $notification['label']; ?></div>
                                                        <div class="text-sm text-gray-500"><?php echo $notification['description']; ?></div>
                                                    </div>
                                                    <label class="flex items-center">
                                                        <input type="checkbox" checked class="mr-2 text-amber-600 focus:ring-amber-500">
                                                        <span class="text-sm">Enable</span>
                                                    </label>
                                                </div>
                                            <?php endforeach; ?>
                                        </div>
                                    </div>

                                    <div>
                                        <h3 class="text-lg font-medium text-gray-900 mb-4">SMS Notifications</h3>
                                        <div class="space-y-3">
                                            <?php
                                            $sms_notifications = [
                                                ['label' => 'Booking Confirmations', 'description' => 'Get SMS confirmation for your reservations'],
                                                ['label' => 'Last-minute Changes', 'description' => 'Receive SMS alerts for any changes to your booking'],
                                                ['label' => 'Arrival Reminders', 'description' => 'Get reminded 2 hours before your reservation time']
                                            ];
                                            foreach ($sms_notifications as $notification):
                                            ?>
                                                <div class="flex items-center justify-between p-4 border border-gray-200 rounded-lg">
                                                    <div>
                                                        <div class="font-medium text-gray-900"><?php echo $notification['label']; ?></div>
                                                        <div class="text-sm text-gray-500"><?php echo $notification['description']; ?></div>
                                                    </div>
                                                    <label class="flex items-center">
                                                        <input type="checkbox" checked class="mr-2 text-amber-600 focus:ring-amber-500">
                                                        <span class="text-sm">Enable</span>
                                                    </label>
                                                </div>
                                            <?php endforeach; ?>
                                        </div>
                                    </div>

                                    <div>
                                        <h3 class="text-lg font-medium text-gray-900 mb-4">Communication Preferences</h3>
                                        <div class="space-y-3">
                                            <div>
                                                <label class="block text-sm font-medium text-gray-700 mb-2">Preferred Contact Method</label>
                                                <select class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-amber-500 focus:border-amber-500">
                                                    <option>Email</option>
                                                    <option>SMS</option>
                                                    <option>Phone Call</option>
                                                    <option>In-app notifications only</option>
                                                </select>
                                            </div>
                                            
                                            <div>
                                                <label class="block text-sm font-medium text-gray-700 mb-2">Best Time to Contact</label>
                                                <select class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-amber-500 focus:border-amber-500">
                                                    <option>Any time</option>
                                                    <option>Morning (8 AM - 12 PM)</option>
                                                    <option>Afternoon (12 PM - 6 PM)</option>
                                                    <option>Evening (6 PM - 10 PM)</option>
                                                </select>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </form>
                    </div>
                </div>

                <!-- Account Actions -->
                <div class="bg-white rounded-lg shadow-sm border border-gray-200 p-6">
                    <h3 class="text-lg font-medium text-gray-900 mb-4">Account Actions</h3>
                    <div class="space-y-3">
                        <button onclick="downloadData()" class="w-full text-left p-3 border border-gray-200 rounded-lg hover:bg-gray-50 transition-colors">
                            <div class="font-medium text-gray-900">Download My Data</div>
                            <div class="text-sm text-gray-500">Get a copy of your reservation history and personal data</div>
                        </button>
                        
                        <button onclick="deactivateAccount()" class="w-full text-left p-3 border border-gray-200 rounded-lg hover:bg-gray-50 transition-colors">
                            <div class="font-medium text-gray-900">Deactivate Account</div>
                            <div class="text-sm text-gray-500">Temporarily disable your account while preserving your data</div>
                        </button>
                        
                        <button onclick="deleteAccount()" class="w-full text-left p-3 border border-red-200 rounded-lg hover:bg-red-50 transition-colors text-red-600">
                            <div class="font-medium">Delete Account</div>
                            <div class="text-sm text-red-500">Permanently delete your account and all associated data</div>
                        </button>
                    </div>
                </div>
            </div>
        </main>
    </div>
</div>

<script>
function showTab(tabName) {
    // Hide all tab contents
    const contents = document.querySelectorAll('.tab-content');
    contents.forEach(content => content.classList.add('hidden'));
    
    // Remove active class from all tabs
    const tabs = document.querySelectorAll('[id$="-tab"]');
    tabs.forEach(tab => {
        tab.classList.remove('border-amber-500', 'text-amber-600');
        tab.classList.add('border-transparent', 'text-gray-500');
    });
    
    // Show selected tab content
    document.getElementById(tabName + '-content').classList.remove('hidden');
    
    // Add active class to selected tab
    const activeTab = document.getElementById(tabName + '-tab');
    activeTab.classList.remove('border-transparent', 'text-gray-500');
    activeTab.classList.add('border-amber-500', 'text-amber-600');
}

function downloadData() {
    alert('Data download feature would be implemented here');
}

function deactivateAccount() {
    if (confirm('Are you sure you want to deactivate your account?')) {
        alert('Account deactivation feature would be implemented here');
    }
}

function deleteAccount() {
    if (confirm('Are you sure you want to permanently delete your account? This action cannot be undone.')) {
        alert('Account deletion feature would be implemented here');
    }
}
</script>

<?php include '../includes/footer.php'; ?>