<?php
require_once '../includes/auth.php';
require_once '../config/database.php';

requireCustomer();

$database = new Database();
$db = $database->getConnection();

// Get filter parameters
$selected_date = $_GET['date'] ?? date('Y-m-d');
$selected_time = $_GET['time'] ?? '19:00';
$guest_count = $_GET['guests'] ?? 2;

// Get all tables with availability check
$tables_query = "
    SELECT rt.*,
           CASE 
               WHEN rt.status != 'available' THEN rt.status
               WHEN EXISTS (
                   SELECT 1 FROM reservations r 
                   WHERE r.table_id = rt.id 
                   AND r.reservation_date = :date 
                   AND r.reservation_time = :time 
                   AND r.status IN ('confirmed', 'pending')
               ) THEN 'reserved'
               ELSE 'available'
           END as current_status
    FROM restaurant_tables rt
    WHERE rt.status != 'maintenance'
    ORDER BY rt.table_number ASC
";

$tables_stmt = $db->prepare($tables_query);
$tables_stmt->bindParam(':date', $selected_date);
$tables_stmt->bindParam(':time', $selected_time);
$tables_stmt->execute();
$tables = $tables_stmt->fetchAll(PDO::FETCH_ASSOC);

// Filter tables by guest count
$suitable_tables = array_filter($tables, function($table) use ($guest_count) {
    return $table['capacity'] >= $guest_count;
});

// Get table statistics
$available_count = count(array_filter($suitable_tables, function($table) {
    return $table['current_status'] === 'available';
}));
$reserved_count = count(array_filter($suitable_tables, function($table) {
    return $table['current_status'] === 'reserved';
}));
$occupied_count = count(array_filter($suitable_tables, function($table) {
    return $table['current_status'] === 'occupied';
}));

$page_title = 'Available Tables';
include '../includes/header.php';
?>

<div class="flex h-screen bg-gray-50">
    <?php include '../includes/sidebar.php'; ?>
    
    <div class="flex-1 flex flex-col overflow-hidden lg:ml-0">
        <?php include '../includes/dashboard_header.php'; ?>
        
        <main class="flex-1 overflow-y-auto p-6">
            <div class="space-y-6 fade-in">
                <div class="flex items-center justify-between">
                    <h1 class="text-2xl font-bold text-gray-900">Available Tables</h1>
                    <div class="text-sm text-gray-500">
                        Find the perfect table for your dining experience
                    </div>
                </div>

                <!-- Booking Filters -->
                <div class="bg-white rounded-lg shadow-sm border border-gray-200 p-6">
                    <h3 class="text-lg font-semibold text-gray-900 mb-4">Check Availability</h3>
                    <form method="GET" class="grid grid-cols-1 md:grid-cols-3 gap-4">
                        <div>
                            <label class="block text-sm font-medium text-gray-700 mb-2">Date</label>
                            <input type="date" name="date" value="<?php echo $selected_date; ?>"
                                   min="<?php echo date('Y-m-d'); ?>"
                                   class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-amber-500 focus:border-amber-500">
                        </div>
                        <div>
                            <label class="block text-sm font-medium text-gray-700 mb-2">Time</label>
                            <select name="time" class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-amber-500 focus:border-amber-500">
                                <?php
                                $times = ['17:00', '17:30', '18:00', '18:30', '19:00', '19:30', '20:00', '20:30', '21:00'];
                                foreach ($times as $time):
                                ?>
                                    <option value="<?php echo $time; ?>" <?php echo $time === $selected_time ? 'selected' : ''; ?>>
                                        <?php echo date('g:i A', strtotime($time)); ?>
                                    </option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        <div>
                            <label class="block text-sm font-medium text-gray-700 mb-2">Guests</label>
                            <select name="guests" class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-amber-500 focus:border-amber-500">
                                <?php for ($i = 1; $i <= 8; $i++): ?>
                                    <option value="<?php echo $i; ?>" <?php echo $i == $guest_count ? 'selected' : ''; ?>>
                                        <?php echo $i; ?> <?php echo $i === 1 ? 'Guest' : 'Guests'; ?>
                                    </option>
                                <?php endfor; ?>
                            </select>
                        </div>
                        <div class="md:col-span-3">
                            <button type="submit" class="bg-amber-600 text-white px-6 py-2 rounded-lg hover:bg-amber-700">
                                Check Availability
                            </button>
                        </div>
                    </form>
                </div>

                <!-- Table Availability Summary -->
                <div class="grid grid-cols-2 md:grid-cols-4 gap-4">
                    <div class="bg-white rounded-lg shadow-sm border border-gray-200 p-4 text-center hover-scale">
                        <div class="text-2xl font-bold text-green-600"><?php echo $available_count; ?></div>
                        <div class="text-sm text-gray-500">Available</div>
                    </div>
                    <div class="bg-white rounded-lg shadow-sm border border-gray-200 p-4 text-center hover-scale">
                        <div class="text-2xl font-bold text-yellow-600"><?php echo $reserved_count; ?></div>
                        <div class="text-sm text-gray-500">Reserved</div>
                    </div>
                    <div class="bg-white rounded-lg shadow-sm border border-gray-200 p-4 text-center hover-scale">
                        <div class="text-2xl font-bold text-red-600"><?php echo $occupied_count; ?></div>
                        <div class="text-sm text-gray-500">Occupied</div>
                    </div>
                    <div class="bg-white rounded-lg shadow-sm border border-gray-200 p-4 text-center hover-scale">
                        <div class="text-2xl font-bold text-blue-600"><?php echo count($suitable_tables); ?></div>
                        <div class="text-sm text-gray-500">Suitable Tables</div>
                    </div>
                </div>

                <!-- Tables Grid -->
                <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
                    <?php if (empty($suitable_tables)): ?>
                        <div class="col-span-full text-center py-12">
                            <i data-lucide="users" class="w-12 h-12 text-gray-400 mx-auto mb-4"></i>
                            <h3 class="text-lg font-medium text-gray-900 mb-2">No Suitable Tables Found</h3>
                            <p class="text-gray-500 mb-4">
                                No tables are available for <?php echo $guest_count; ?> guests at <?php echo date('g:i A', strtotime($selected_time)); ?> on <?php echo date('M j, Y', strtotime($selected_date)); ?>.
                            </p>
                            <p class="text-sm text-gray-500">
                                Try adjusting your date, time, or party size to see more options.
                            </p>
                        </div>
                    <?php else: ?>
                        <?php foreach ($suitable_tables as $table): ?>
                            <?php
                            $status_colors = [
                                'available' => 'border-green-200 bg-green-50',
                                'occupied' => 'border-red-200 bg-red-50',
                                'reserved' => 'border-yellow-200 bg-yellow-50',
                                'maintenance' => 'border-gray-200 bg-gray-50'
                            ];
                            $status_icons = [
                                'available' => '🟢',
                                'occupied' => '🔴',
                                'reserved' => '🟡',
                                'maintenance' => '⚪'
                            ];
                            $border_class = $status_colors[$table['current_status']] ?? 'border-gray-200 bg-white';
                            $status_icon = $status_icons[$table['current_status']] ?? '⚪';
                            ?>
                            <div class="bg-white rounded-lg shadow-sm border-2 p-6 transition-all hover:shadow-md hover-scale <?php echo $border_class; ?>">
                                <div class="flex items-center justify-between mb-4">
                                    <div class="flex items-center">
                                        <span class="text-2xl mr-2"><?php echo $status_icon; ?></span>
                                        <h3 class="text-xl font-semibold text-gray-900">Table <?php echo $table['table_number']; ?></h3>
                                    </div>
                                    <?php
                                    $status_badge_colors = [
                                        'available' => 'bg-green-100 text-green-800',
                                        'occupied' => 'bg-red-100 text-red-800',
                                        'reserved' => 'bg-yellow-100 text-yellow-800',
                                        'maintenance' => 'bg-gray-100 text-gray-800'
                                    ];
                                    $badge_color = $status_badge_colors[$table['current_status']] ?? 'bg-gray-100 text-gray-800';
                                    ?>
                                    <span class="inline-flex px-2 py-1 text-xs font-semibold rounded-full <?php echo $badge_color; ?>">
                                        <?php echo ucfirst($table['current_status']); ?>
                                    </span>
                                </div>

                                <div class="space-y-3 mb-4">
                                    <div class="flex items-center text-gray-600">
                                        <i data-lucide="users" class="w-5 h-5 mr-2"></i>
                                        <span>Seats up to <?php echo $table['capacity']; ?> people</span>
                                    </div>
                                    
                                    <div class="flex items-center text-gray-600">
                                        <i data-lucide="map-pin" class="w-5 h-5 mr-2"></i>
                                        <span><?php echo htmlspecialchars($table['location']); ?></span>
                                    </div>

                                    <div class="flex items-center text-gray-600">
                                        <i data-lucide="calendar" class="w-5 h-5 mr-2"></i>
                                        <span><?php echo date('M j, Y', strtotime($selected_date)); ?></span>
                                    </div>

                                    <div class="flex items-center text-gray-600">
                                        <i data-lucide="clock" class="w-5 h-5 mr-2"></i>
                                        <span><?php echo date('g:i A', strtotime($selected_time)); ?></span>
                                    </div>
                                </div>

                                <!-- Table Features -->
                                <div class="mb-4">
                                    <h4 class="text-sm font-medium text-gray-700 mb-2">Features:</h4>
                                    <div class="space-y-1">
                                        <?php if ($table['location'] === 'Window Side'): ?>
                                            <div class="flex items-center text-sm text-gray-600">
                                                <i data-lucide="check" class="w-3 h-3 mr-1 text-green-500"></i>
                                                Window view
                                            </div>
                                        <?php endif; ?>
                                        <?php if ($table['location'] === 'VIP Section'): ?>
                                            <div class="flex items-center text-sm text-gray-600">
                                                <i data-lucide="check" class="w-3 h-3 mr-1 text-green-500"></i>
                                                Premium location
                                            </div>
                                        <?php endif; ?>
                                        <?php if ($table['location'] === 'Private Room'): ?>
                                            <div class="flex items-center text-sm text-gray-600">
                                                <i data-lucide="check" class="w-3 h-3 mr-1 text-green-500"></i>
                                                Private dining
                                            </div>
                                        <?php endif; ?>
                                        <div class="flex items-center text-sm text-gray-600">
                                            <i data-lucide="check" class="w-3 h-3 mr-1 text-green-500"></i>
                                            Professional service
                                        </div>
                                    </div>
                                </div>

                                <!-- Booking Button -->
                                <?php if ($table['current_status'] === 'available'): ?>
                                    <button onclick="bookTable(<?php echo $table['id']; ?>, '<?php echo $selected_date; ?>', '<?php echo $selected_time; ?>', <?php echo $guest_count; ?>)" 
                                            class="w-full bg-amber-600 text-white py-2 px-4 rounded-lg hover:bg-amber-700 transition-colors font-medium">
                                        Book This Table
                                    </button>
                                <?php elseif ($table['current_status'] === 'reserved'): ?>
                                    <button disabled class="w-full bg-yellow-100 text-yellow-800 py-2 px-4 rounded-lg cursor-not-allowed font-medium">
                                        Reserved for <?php echo date('g:i A', strtotime($selected_time)); ?>
                                    </button>
                                <?php else: ?>
                                    <button disabled class="w-full bg-gray-100 text-gray-500 py-2 px-4 rounded-lg cursor-not-allowed font-medium">
                                        Not Available
                                    </button>
                                <?php endif; ?>
                            </div>
                        <?php endforeach; ?>
                    <?php endif; ?>
                </div>

                <!-- Legend -->
                <div class="bg-white rounded-lg shadow-sm border border-gray-200 p-4">
                    <h4 class="font-medium text-gray-900 mb-3">Table Status Legend</h4>
                    <div class="grid grid-cols-2 md:grid-cols-4 gap-4 text-sm">
                        <div class="flex items-center">
                            <span class="text-lg mr-2">🟢</span>
                            <span>Available for booking</span>
                        </div>
                        <div class="flex items-center">
                            <span class="text-lg mr-2">🟡</span>
                            <span>Already reserved</span>
                        </div>
                        <div class="flex items-center">
                            <span class="text-lg mr-2">🔴</span>
                            <span>Currently occupied</span>
                        </div>
                        <div class="flex items-center">
                            <span class="text-lg mr-2">⚪</span>
                            <span>Under maintenance</span>
                        </div>
                    </div>
                </div>
            </div>
        </main>
    </div>
</div>

<script>
function bookTable(tableId, date, time, guests) {
    if (confirm('Book this table for ' + date + ' at ' + time + ' for ' + guests + ' guests?')) {
        // Redirect to reservations page with pre-filled data
        window.location.href = '/customer/reservations.php?table=' + tableId + '&date=' + date + '&time=' + time + '&guests=' + guests;
    }
}
</script>

<?php include '../includes/footer.php'; ?>